from datetime import datetime
from logging import getLogger
from urllib.parse import urljoin

from django.conf import settings
from django.utils.dateparse import parse_datetime
from django.utils.functional import cached_property
from django.utils.timezone import get_current_timezone

from intranet.search.core.swarm.api_indexer import PagedApiIndexer
from requests.exceptions import RequestException

from intranet.search.core.sources.utils import date_as_factor, date_to_timestamp
from intranet.search.core.utils import http


log = getLogger(__name__)


class ConductorApiClient:
    base_api = settings.ISEARCH['api']['conductor']['api_v2']

    @cached_property
    def session(self):
        session = http.create_session()
        session.headers.update(self.base_api.headers())
        return session

    def get_api_url(self, obj):
        """ Возвращает url объекта в новом /api/v2
        """
        return obj['links']['related']

    def fetch(self, url, **kwargs):
        response = http.call_with_retry(self.session.get, url, **kwargs)
        return response.json()

    def fetch_page(self, url, page, per_page=20, start_date=None, end_date=None):
        """ Извлекает одну из страниц коллекции
        """
        params = {'page[number]': page, 'page[size]': per_page}
        if start_date:
            end_date = end_date or datetime.utcnow()
            params['filter[updated_at]'] = '{}..{}'.format(start_date.isoformat(),
                                                           end_date.isoformat())
        return self.fetch(url, params=params)

    def pages_count(self, data):
        return data['meta']['page_count']

    def fetch_all(self, url):
        """ Извлекает все страницы коллекции объектов
        """
        page = 1
        pages = 1
        items = []
        while page <= pages:
            result = self.fetch_page(url, page)
            items.extend(result['data'])
            page += 1
            pages = self.pages_count(result)
        return items


class SourceBase(PagedApiIndexer):
    """ Базовый индексатор для кондуктора
    """
    # название источника для индексации, переопределяется в наследниках
    source = ''
    # человекочитаемые нахвания типа источника
    source_label = ''
    source_label_plural = ''
    # префикс для названий статический факторов
    stat_factor_prefix = ''
    # базовый урл кондуктора
    base_url = settings.ISEARCH['api']['conductor']['endpoint'].url()
    # список языков для сниппета
    snippet_languages = ('ru', )

    # дата обновления объекта по умолчанию, отправляется в документ,
    # если получили объект без дат
    default_updated = get_current_timezone().localize(datetime.fromtimestamp(0))
    snippet_class = None

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.api_client = ConductorApiClient()
        self.source_url = settings.ISEARCH['api']['conductor'][self.source].url()

    def fetch_page(self, page=1):
        start_date = None
        if self.options['ts']:
            start_date = datetime.utcfromtimestamp(float(self.options['ts']))
        return self.api_client.fetch_page(self.source_url, page, start_date=start_date)

    def fetch_objects(self, page=1):
        return (obj for obj in self.fetch_page(page)['data'])

    def get_counts(self):
        pages_count = self.api_client.pages_count(self.fetch_page())
        return pages_count, None

    def get_doc_updated(self, obj):
        date = obj['attributes']['updated_at'] or obj['attributes']['created_at'] or ''
        if not date:
            log.warning('Got object without date: {}, {}'.format(obj['type'], obj['id']))
        return parse_datetime(date) or self.default_updated

    def emit_factors(self, doc, obj):
        """ Добавление факторов к документу
        """
        if doc.updated != self.default_updated:
            doc.emit_factor('objectUpdated', date_as_factor(doc.updated))
        if obj['attributes']['created_at']:
            doc.emit_factor('objectCreated', date_as_factor(date_to_timestamp(obj['attributes']['created_at'])))

        doc.emit_factor('is%s' % self.stat_factor_prefix.title(), 1)

    def emit_attrs(self, doc, obj):
        """ Добавление атрибутов к документу
        """
        if doc.updated != self.default_updated:
            doc.emit_facet_attr('object_updated', doc.updated.strftime('%Y-%m'),
                                label=doc.updated.strftime('%Y-%m'))
            doc.emit_sort_attr('updated', int(doc.updated.strftime('%s')))
        doc.emit_facet_attr('object_type', self.source,
                            label=self.source_label.title(), label_en=self.source.title())

        if obj['attributes']['created_at']:
            created_facet = parse_datetime(obj['attributes']['created_at']).strftime("%Y-%m")
            doc.emit_facet_attr("object_created", created_facet, label=created_facet)

    def create_snippet(self, obj, lang='ru', **kwargs):
        data = obj['attributes']
        snippet = {
            'url': self.get_doc_url(obj),
            'title': self.get_obj_name(obj),
            'description': data.get('description') or '',
            'created': parse_datetime(data['created_at'] or ''),
            'updated': parse_datetime(data['updated_at'] or ''),
        }
        snippet["breadcrumbs"] = [
            {"url": urljoin(self.base_url, self.source),
             "name": self.source_label_plural.title()},
            {"url": self.get_doc_url(obj), "name": self.get_obj_name(obj)}
        ]
        return self.snippet_class(snippet)

    def get_obj_name(self, obj):
        return obj['attributes']['name']

    def get_doc_url(self, obj):
        return obj['attributes'].get('url')

    def fetch_related(self, obj, related_name, one=False):
        url = self.api_client.get_api_url(obj['relationships'][related_name])
        try:
            if one:
                return self.api_client.fetch(url)
            else:
                return self.api_client.fetch_all(url)
        except RequestException:
            log.warning('Cannot get related object: %s', url)
            return None

    def fetch_related_one(self, obj, related_name):
        result = self.fetch_related(obj, related_name, one=True)
        return result['data'] if result else None

    def dump_related(self, related):
        url = self.get_doc_url(related)
        if not url:
            return related['attributes']['name']
        return {'title': related['attributes']['name'], 'url': url}
