from logging import getLogger

from intranet.search.core.snippets.conductor import ProjectsSnippet
from intranet.search.core.sources.conductor.base import SourceBase
from intranet.search.core.sources.utils import get_by_lang, Services

log = getLogger(__name__)


class Source(SourceBase):
    """ Индексатор проектов
    """
    source = 'projects'
    source_label = 'проект'
    source_label_plural = 'проекты'
    stat_factor_prefix = 'project'
    need_fetch = True
    snippet_class = ProjectsSnippet

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.services_cache = Services(self.cache_storage)

    def do_fetch(self, obj, **kwargs):
        kwargs = {}
        abc_service_id = obj['attributes']['abc_service_id']
        if abc_service_id:
            try:
                kwargs['service_info'] = self.services_cache[abc_service_id]
            except Exception:
                log.exception('Cannot get abc service details: %s', abc_service_id)

        self.next('create', obj=obj, **kwargs)

    def create_body(self, obj, **kwargs):
        data = obj['attributes']
        body = {
            'description': data['description'] or '',
            'hidden': {
                'name': data['name'],
                'emails': {
                    'mailto': data['mailto'] or '',
                    'root_email': data['root_email'] or '',
                    'testing_email': data['testing_mail'] or '',
                },
                'service_info': {
                    'service_name': kwargs.get('service_info', {}).get('name', '')
                }
            }
        }
        return body

    def create_snippet(self, obj, lang='ru', **kwargs):
        snippet = super().create_snippet(obj, lang, **kwargs)
        data = obj['attributes']
        snippet['mailto'] = data['mailto']
        snippet['root_email'] = data['root_email']
        snippet['testing_email'] = data['testing_mail']

        if kwargs.get('service_info'):
            snippet['service_name'] = get_by_lang(kwargs['service_info']['name'], lang)
        return snippet
