from intranet.search.core.snippets.directory import GroupSnippet
from intranet.search.core.sources.directory.base import SourceBase
from intranet.search.core.sources.directory.client import join_fields, GROUP_GENERIC_TYPE
from intranet.search.core.sources.utils import get_by_lang, get_document_url, get_person_name_by_lang, \
    hash_factor


class Source(SourceBase):
    """ Индексатор групп директории
    """
    doc_type = 'group'

    _group_fields = (
        'id',
        'name',
        'label',
        'description',
        'email',
        'type',
        'external_id',
        'aliases',
    )
    _admin_fields = (
        'id',
        'nickname',
        'name',
    )

    FIELDS = join_fields((
        join_fields(_group_fields),
        join_fields(_admin_fields, prefix='admins'),
    ))

    def _request_directory(self, page=1):
        """ Не возвращает ничего, начиная с ISEARCH-7356
        """
        params = {'page': page, 'fields': self.FIELDS, 'type': GROUP_GENERIC_TYPE}
        return self.api_client.get_groups(self.org_directory_id, **params)

    def fetch_object(self, obj_id):
        return self.api_client.get_group(obj_id, self.org_directory_id, fields=self.FIELDS)

    def do_fetch(self, obj, **kwargs):
        # дергаем директорию, чтобы получить список админов
        obj = self.fetch_object(obj['id'])
        self.next('create', obj=obj)

    def create_body(self, obj, **kwargs):
        body = {
            'name': obj['name'],
            'description': obj['description'],
            'label': obj['label'],
            'email': obj['email'],
            'type': obj['type'],
            'external_id': obj['external_id'],
        }
        return body

    def emit_factors(self, doc, obj):
        doc.emit_factor('isGroup', 1)
        doc.emit_factor('urlHash', hash_factor(doc.url))

    def emit_attrs(self, doc, obj):
        if obj['label']:
            doc.emit_suggest_attr(obj['label'])
        for lang in ('ru', 'en'):
            name = get_by_lang(obj['name'], lang)
            if name:
                doc.emit_suggest_attr(name)
        for alias in obj['aliases'] or []:
            doc.emit_suggest_attr(alias)

    def create_snippet(self, obj, lang='ru', **kwargs):
        data = {
            'doc_type': self.doc_type
        }
        for field in ('id', 'label', 'email', 'type'):
            data[field] = obj[field]
        for field in ('name', 'description'):
            data[field] = get_by_lang(obj[field], lang)

        data['admins'] = []
        for admin in obj.get('admins', []):
            data['admins'].append({
                'uid': admin['id'],
                'login': admin['nickname'],
                'name': get_person_name_by_lang(admin, lang),
                'url': get_document_url('users').format(login=admin['nickname']),
            })
        return GroupSnippet(data)

    def do_push(self, data, delete=False, **kwargs):
        if data.get('type') and data['type'] != GROUP_GENERIC_TYPE:
            return
        super().do_push(data, delete)
