from intranet.search.core.utils import get_ids_repository

from intranet.search.core.swarm import Indexer


class Source(Indexer):
    """ Индексатор оборудования
    """
    equipment_repo = get_ids_repository('staff', 'equipment')

    types_data = {
        'printer': {'id': 0, 'ru': 'Принтер', 'en': 'Printer'},
        'scanner': {'id': 1, 'ru': 'Сканер', 'en': 'Scanner'},
        'copier': {'id': 2, 'ru': 'Копировальный аппарат', 'en': 'Copier'},
        'mfp': {'id': 3, 'ru': 'МФП', 'en': 'MFP'},
        'fax': {'id': 7, 'ru': 'Факс', 'en': 'Fax'},
    }

    def do_walk(self, **kwargs):
        equipment_items = self.equipment_repo.getiter(
            lookup={'type': ','.join(self.types_data.keys())})

        for item in equipment_items:
            self.next('create', obj=item)

    def do_create(self, obj, **kwargs):
        search_url = "http://staff.yandex-team.ru/map/#/equipment/%s" % obj['id']

        doc = self.create_document(search_url)

        if obj['is_deleted'] or obj['floor']['office']['is_deleted']:
            self.next('store', document=doc, delete=True)
            return

        if 'id' in obj['floor']:
            doc.emit_group_attr('floor', int(obj['floor']['id']))
            if 'id' in obj['floor']['office']:
                doc.emit_group_attr('office', int(obj['floor']['office']['id']))

        for lang in ('ru', 'en'):
            doc.emit_snippet(self.create_snippet(obj, lang), lang)

        body = {'name': obj['name'],
                'name_dns': obj['hostname'],
                'description': obj['description'],
                'types': {'ru': self.types_data[obj['type']]['ru'],
                          'en': self.types_data[obj['type']]['en']},
                'place': {'floor': obj['floor'].get('name'),
                          'office': obj['floor']['office'].get('name')}}
        doc.emit_body(body)

        self.next('store', document=doc)

    def create_snippet(self, obj, lang='ru'):
        snippet = {
            'id': obj['id'],
            'name': obj['name'],
            'name_dns': obj['hostname'],
            'description': obj['description'],
            'coord_x': obj['location']['x'],
            'coord_y': obj['location']['y'],
            'floor_id': obj['floor']['id'],
            'type_id': self.types_data[obj['type']]['id'],
            'type_ru': self.types_data[obj['type']]['ru'],
            'type_en': self.types_data[obj['type']]['en'],
            'type': obj['type'],
        }

        if 'name' in obj['floor']:
            obj['floor_name'] = obj['floor']['name'][lang]

        if 'id' in obj['floor']['office']:
            snippet['office_id'] = obj['floor']['office']['id']
            snippet['office_name'] = obj['floor']['office']['name'][lang]

        if obj['floor']['office']['city']:
            snippet['city_id'] = obj['floor']['office']['city']['id']
            snippet['city_name'] = obj['floor']['office']['city']['name'][lang]

        return snippet
