from datetime import date

from django.utils.dateparse import parse_datetime
from math import floor


from intranet.search.core.snippets.femida import CandidateSnippet
from intranet.search.core.sources.femida.base import SourceBase
from intranet.search.core.sources.people.utils import gen_phones
from intranet.search.core.sources.utils import date_as_factor, date_to_timestamp, hash_factor, normalize


class Source(SourceBase):
    list_api = 'candidates'
    item_api = 'candidate'
    item_url = 'candidate_html'
    page_size = 1000

    def parse_keys(self):
        result = {}
        for key in self.options['keys']:
            if ':' in key:
                name, value = key.split(':', 1)
                result[name] = value
            else:
                result.setdefault('id', [])
                result['id'].append(key)
        return result

    def do_create(self, result, **kwargs):
        url = self.get_doc_url(result)
        doc = self.create_document(url, updated=parse_datetime(result['modified']))
        doc.emit_body(self.create_body(result))
        doc.emit_snippet(self.create_snippet(result))
        self.emit_attrs(doc, result)
        self.emit_factors(doc, result)
        self.next('store', document=doc, body_format='json')

    def create_body(self, obj):
        contacts = []
        for c in obj['contacts']:
            if c['type'] == 'phone':
                value = ' '.join(gen_phones(c['account_id']))
            else:
                value = c['account_id']
            contacts.append('{} {}'.format(c['type'], value))

        institutions = []
        for edu in obj['educations']:
            institutions.append(edu['institution'])
        employers = []
        for job in obj['jobs']:
            employers.append(job['employer'])

        body = {
            # зона, в которрой по умолчанию будет происходить поиск
            # (см. request_builders.CandidateSearch)
            'z_base': {
                'id': obj['id'],
                'name': {
                    'z_first_name': obj['first_name'],
                    'z_middle_name': obj['middle_name'],
                    'z_last_name': obj['last_name'],
                },
                'contacts': contacts,
                'login': obj['login'],
                'attachments': [a['text'] for a in obj.get('attachments', []) if a['text']],
            },
            # Зоны для дополнительного поиска по зонам.
            # Названия передаем явно с z_, чтобы форматтер сообщения их не изменил.
            'z_city': obj['city'],
            'z_institution': institutions,
            'z_employer': employers,
        }
        return body

    def create_snippet(self, obj):
        data = {}
        for f in ('id', 'first_name', 'last_name', 'middle_name', 'login', 'city',
                  'responsibles', 'vacancies', 'skills', 'is_duplicate',
                  'professions', 'applications', 'nohire_interviews_count',
                  'hire_interviews_count', 'hire_interviews_avg_grade',
                  'skype_interviews_count', 'skype_interviews_avg_grade',
                  'on_site_interviews_count', 'on_site_interviews_avg_grade',
                  'modified', 'created', 'is_current_employee', 'status', 'tags',
                  'target_cities'):
            data[f] = obj.get(f)

        data['contacts'] = []
        for c in obj['contacts']:
            contact_data = {}
            for f in ('id', 'type', 'account_id', 'is_main', 'is_active'):
                contact_data[f] = c[f]
            data['contacts'].append(contact_data)
        return CandidateSnippet(data)

    def emit_attrs(self, doc, obj):
        # атрибуты для ограничения поиска по acl
        for field in ('is_duplicate', 'is_hidden', 'is_locked', 'has_only_hidden_vacancies',
                      'is_current_employee'):
            doc.emit_search_attr(f'i_{field}', int(obj.get(field, False)))

        doc.emit_search_attr('i_modified', date_to_timestamp(obj['modified']))
        doc.emit_search_attr('i_created', date_to_timestamp(obj['created']))

        # Фемиде нужно искать по логину как "s_login НЕ равен ...". При этом у многих
        # кандидатов логины пустые, а такие атрибуты не сохраняются в саас. Чтобы кандидаты
        # с пустыми логинами находились по запросу, добавляем им фейковое значение атрибута.
        doc.emit_search_attr('s_login', obj['login'] or '_')

        vacancy_access = set()
        for vacancy in obj['vacancies']:
            vacancy_access.update(vacancy.get('access', []))
        for access in vacancy_access:
            doc.emit_search_attr('s_vacancy_access', access)

        for sphere in obj.get('professional_spheres', []):
            doc.emit_search_attr('i_professional_sphere', sphere['id'])
        for interview in obj.get('interviews', []):
            doc.emit_search_attr('s_interviewer', interview['interviewer'])

        # фасеты и дополнительные атрибуты поиска
        for skill in obj.get('skills', []):
            doc.emit_facet_attr('skill', skill['id'], label=skill['name'])
        for profession in obj.get('professions', []):
            doc.emit_facet_attr('profession', profession['id'], label=profession['name'])
        for responsible in obj.get('responsibles', []):
            doc.emit_facet_attr('responsible', responsible, label=responsible)
        for tag in obj.get('tags', []):
            doc.emit_facet_attr('tag', tag, label=tag)
        for city in obj.get('target_cities', []):
            doc.emit_facet_attr('target_city', city['id'],
                                label=city['name_ru'], label_en=city['name_en'])
        if obj.get('status'):
            doc.emit_facet_attr('status', obj['status'], label=obj['status'])

        for field in ('nohire_interviews_count',
                      'hire_interviews_count',
                      'on_site_interviews_count'):
            if obj[field] is not None:
                doc.emit_search_attr(f'i_{field}', int(obj[field]))

        for field in ('skype_interviews_avg_grade',
                      'hire_interviews_avg_grade',
                      'on_site_interviews_avg_grade'):
            if obj[field] is not None:
                doc.emit_search_attr(f'i_{field}', int(floor(obj[field])))

    def emit_factors(self, doc, obj):
        doc.emit_factor('updated', date_as_factor(doc.updated, date(2050, 1, 1)))
        doc.emit_factor('urlHash', hash_factor(doc.url))

        doc.emit_factor('skypeInterviewsAvgGrade', normalize(obj['skype_interviews_avg_grade'], 8))
        doc.emit_factor('onSiteInterviewsAvgGrade', normalize(obj['on_site_interviews_avg_grade'], 8))
        doc.emit_factor('noHire', normalize(obj['nohire_interviews_count'], 1))
        doc.emit_factor('inProgress', float(obj['status'] == 'in_progress'))
