import logging

from django.conf import settings

import ids

from intranet.search.core.swarm import Indexer
from intranet.search.core.utils import http, get_ids_repository
from intranet.search.core.sources.utils import get_by_lang, get_document_url, get_person_fio

from intranet.search.core.snippets.goals import GoalsSnippet


log = logging.getLogger(__name__)

dep_repo = get_ids_repository('staff', 'group')


status_map = {
    0: 'по плану',
    1: 'есть риски',
    2: 'заблокирована',
    3: 'отмена',
    4: 'достигнута',
    5: 'новая',
}


class Source(Indexer):
    def __init__(self, options):
        super().__init__(options)
        self.session = http.create_session(max_retries=3)

    def do_walk(self, url=None, **kwargs):
        url = url or settings.ISEARCH['api']['goals']['goals'].url()

        response = http.call_with_retry(self.session.get, url,
                                        headers=settings.ISEARCH['api']['goals']['goals'].headers(),
                                        verify=False)

        result = response.json()

        for goal in result['results']:
            if not goal['is_confidential']:
                self.next('fetch', data=goal)

        if result['next']:
            self.next('walk', url=result['next'])

    def do_fetch(self, data=None, **kwargs):
        try:
            lookup = {
                'id': data['department_id'],
                'type': 'department',
                'is_deleted': 'True,False',
            }
            department = dep_repo.get_one(lookup=lookup)
        except ids.exceptions.BackendError as exc:
            if hasattr(exc, 'response') and exc.response.status_code == 404:
                department = None
                log.error("Can't get dep data for goal %s", data['id'])
            else:
                raise exc

        self.next('create', data=data, department=department)

    def do_create(self, data=None, department=None, **kwargs):
        url = get_document_url('goal').format(id=data['id'])

        doc = self.create_document(url)

        doc.emit_facet_attr('status', data['status'], status_map.get(data['status']))

        if department:
            department = department['department']
            doc.emit_facet_attr('department', department['url'],
                                department['name']['short']['ru'].strip() or department['name']['full']['ru'],
                                department['name']['short']['en'].strip() or department['name']['full']['en'])

        if data['responsible']:
            doc.emit_facet_attr('responsible', data['responsible']['login'],
                                get_person_fio(data['responsible'], 'ru', False),
                                get_person_fio(data['responsible'], 'en'))

        for customer in data['customers']:
            doc.emit_facet_attr('customer', customer['person']['login'],
                                get_person_fio(customer['person'], 'ru', False),
                                get_person_fio(customer['person'], 'en'))

        for imp in data['implementers']:
            doc.emit_facet_attr('implementer', imp['person']['login'],
                                get_person_fio(imp['person'], 'ru', False),
                                get_person_fio(imp['person'], 'en'))

        for tag in data['tags']:
            doc.emit_facet_attr('tag', tag['id'], tag['text'])

        # TODO: uncomment when necessary
        # self.emit_factor('importance', data['importance'])
        # self.emit_factor('status', data['status'])
        # self.emit_factor('deadline', date_as_factor(data['deadline']))

        self.create_goal_body(doc, data, department)
        self.create_goal_snippet(doc, data, department, url, 'ru')
        self.create_goal_snippet(doc, data, department, url, 'en')

        self.next('store', document=doc, body_format='json')

    def create_goal_body(self, doc, data, department):
        def get_person_data(person):
            return (person['name'], person['login']) if person else ''

        def get_person_list_data(key):
            return [get_person_data(p['person']) for p in data[key]]

        body = {
            'title': data['title'],
            'description': data['description'],
            'comment': data['comment'],
            'z_ns_hidden': {
                'implementers': get_person_list_data('implementers'),
                'customers': get_person_list_data('customers'),
                'tags': [tag['text'] for tag in data['tags']],
                'comments': [com['text'] for com in data['comments']],
                'responsible': get_person_data(data['responsible']),
            }
        }

        if department:
            body['z_ns_hidden'].update({
                'department': department['name'],
                'department_url': department['url'],
            })

        doc.emit_body(body)

    def create_goal_snippet(self, doc, data, department, url, lang):
        if status_map.get(data['status']):
            title = '{} [{}]'.format(data['title'], status_map[data['status']])
        else:
            title = data['title']

        snippet = {
            'url': url,
            'title': title,
            'description': data['description'],
            'comment': data['comment'],
            'deadline': data['deadline'],
            'importance': data['importance'],
            'status': status_map.get(data['status']),
            'breadcrumbs': [],
        }

        if data['responsible']:
            snippet['person'] = {
                'login': data['responsible']['login'],
                'first_name': get_by_lang(data['responsible']['name']['first'], lang),
                'last_name': get_by_lang(data['responsible']['name']['last'], lang),
            }

        if department:
            snippet['breadcrumbs'] = [{
                'name': (
                    get_by_lang(department['name']['short'], lang).strip()
                    or get_by_lang(department['name']['full'], lang)
                ),
                'url': get_document_url('department').format(url=department['url']),
            }]

        doc.emit_snippet(GoalsSnippet(snippet), lang)
