from logging import getLogger

from django.utils.dateparse import parse_datetime

from intranet.search.core.utils import get_ids_repository
from intranet.search.core.sources.idm.base import SourceBase
from intranet.search.core.sources.utils import get_by_lang, date_as_factor, get_suggest_parts, get_document_url
from intranet.search.core.snippets.idm import GroupSnippet


log = getLogger(__name__)


class Source(SourceBase):
    """ Индексатор нод для групп idm """

    source = 'groups'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.groups_repo = get_ids_repository('staff', 'group', timeout=5)

    def fetch_objects(self, start_id=None):
        params = {
            '_fields': 'id,name,url,service,type,ancestors,_meta.modified_at',
            '_limit': 100,
            'is_deleted': 'false,true',
        }
        if self.options['keys']:
            params['url'] = ','.join(self.options['keys'])
        if start_id is not None:
            params['id'] = start_id
        return self.groups_repo.getiter(params)

    def get_doc_updated(self, obj):
        return parse_datetime(obj['_meta']['modified_at'])

    def get_doc_url(self, obj):
        return get_document_url('groups').format(slug=obj['url'])

    def create_body(self, obj, **kwargs):
        return {
            'id': obj['id'],
            'name': obj['name'],
            'url': obj['url'],
        }

    def do_fetch(self, obj, **kwargs):
        self.next('create', obj=obj)

    def emit_attrs(self, doc, obj):
        doc.emit_suggest_attr(str(obj['id']))
        doc.emit_suggest_attr(obj['url'])
        if type(obj['name']) is str:
            doc.emit_suggest_attr(obj['name'])
        elif type(obj['name']) is dict:
            for suggest_element in obj['name'].values():
                doc.emit_suggest_attr(suggest_element)
                for part in get_suggest_parts(suggest_element):
                    doc.emit_suggest_attr(part)

        doc.emit_search_attr('s_group_id', str(obj['id']))
        doc.emit_search_attr('s_type', 'group')

    def emit_factors(self, doc, obj):
        doc.emit_factor('groupUpdated', date_as_factor(doc.updated))
        doc.emit_factor('isGroup', 1)
        group_type = obj['type']

        # добавляем группировочные атрибуты для подсчета пользовательских факторов
        if group_type == 'department':
            doc.emit_factor('isDepartment', 1)
            if obj['url'] == 'yandex':
                doc.emit_factor('departmentIsYandex', 1)

            doc.emit_group_attr('department', int(obj['id']))
            if obj.get('ancestors'):
                parent_id = obj['ancestors'][-1]['id']
                doc.emit_group_attr('parent_department', parent_id)
        elif group_type == 'service':
            doc.emit_factor('isService', 1)
            doc.emit_group_attr('service', int(obj['id']))
        elif group_type == 'servicerole':
            doc.emit_factor('isServicerole', 1)
            if obj.get('ancestors'):
                service_id = obj['ancestors'][-1]['id']
                doc.emit_group_attr('service', service_id)
        elif group_type == 'wiki':
            doc.emit_factor('isWiki', 1)

    def create_snippet(self, obj, lang='ru', **kwargs):
        data = {
            'slug': obj['url'],
            'url': self.get_doc_url(obj),
            'title': get_by_lang(obj['name'], lang),
            'object_id': obj['id'],
            'object_type': self.source,
            'type': obj['type'],
            'ancestors': []
        }
        if obj.get('ancestors'):
            data['ancestors'] = [{'id': a['id'], 'name': a['name']}
                                 for a in obj['ancestors']]
        return GroupSnippet(data)
