from dateutil.parser import parse

from intranet.search.core.sources.idm.utils import gen_options_by_special_chars
from intranet.search.core.sources.utils import (
    get_by_lang,
    timestamp_to_utc_date,
    get_suggest_parts,
    date_to_timestamp,
)
from intranet.search.core.snippets.idm import RolenodeSnippet
from intranet.search.core.sources.idm.base import SourceBase


ALIASES_MAX_COUNT_FOR_SUGGEST = 100


class Source(SourceBase):
    """ Индексатор нод idm
    """
    source = 'rolenodes'

    def do_create(self, obj, **kwargs):
        if obj['state'] == 'deprived':
            self.do_delete(obj, delete_children=False)
        else:
            obj['parent_path'] = obj['parent_path'] or '/'
            super().do_create(obj, **kwargs)

    def do_delete(self, obj, delete_children=True, **kwargs):
        doc = self.create_document(self.get_doc_url(obj), updated=self.get_doc_updated(obj))
        self.next('store', document=doc, delete=True)

        if delete_children:
            # помимо самого узла удаляем всех его потомков
            parent_path = self.get_doc_id(obj)
            for st in self.document_storages:
                st.delete_by_filter(f's_parent_path:"{parent_path}*"')

    def fetch_page(self, page=1, **kwargs):
        kwargs.setdefault('per_page', 250)
        kwargs.setdefault('is_key', 0)
        kwargs.setdefault('is_public', 1)
        kwargs.setdefault('state', 'all')

        kwargs.update(self.parse_keys())

        if self.options['ts']:
            kwargs['updated__since'] = timestamp_to_utc_date(self.options['ts']).isoformat()

        return self.api_client.fetch_page(self.source, page, **kwargs)

    def parse_keys(self):
        """ В ключах можно передать просто систему для индексации
        (в том числе в формате system:<system_slug>, генерируемом из фильтров при приёме пушей)
        или путь к ноде в виде: /:system_slug/:slug_path/
        """
        result = {}
        if self.options['keys']:
            obj_path = self.options['keys'][0].lstrip('/')
            if ':' in obj_path:
                name, value = obj_path.split(':', 1)
                if name == 'system':
                    value = value.rstrip('/')
                result[name] = value
            else:
                parts = obj_path.split('/', 1)
                result['system'] = parts[0]
                if len(parts) > 1:
                    result['slug_path'] = '/' + parts[1]

            for key in self.options['keys'][1:]:
                name, value = key.split(':', 1)
                result[name] = value
        return result

    def objects_count(self, updated_since=None, updated_till=None):
        updated_till.replace(microsecond=0)
        updated_since.replace(microsecond=0)
        page = self.fetch_page(updated__since=updated_since.isoformat(),
                               updated__until=updated_till.isoformat(),
                               per_page=1)
        return page['meta']['total_count']

    def get_doc_url(self, obj):
        return f'{self.source}_id:{self.get_doc_id(obj)}'

    def get_doc_updated(self, obj):
        return parse(obj['updated_at']) if obj.get('updated_at') else None

    def create_body(self, obj, **kwargs):
        body = {
            'system': {
                'name': obj['system']['name'],
                'slug': obj['system']['slug']
            },
            'slug': obj['slug'],
            'name': obj['name'],
            'description': {
                'human_short': obj['human_short'],
                'help': obj['help'],
            },
            'url': obj['slug_path'],
            'parent_path': obj['parent_path'],
            'rolenodes_aliases': '',
        }

        aliases = set()
        for alias in obj['aliases']:
            for value in alias['name'].values():
                # у алиасов формат вида {"type": "default", "name": {"ru": "...", "en": "..."}}
                aliases.add(value)
        body['rolenodes_aliases'] = ' '.join(aliases)

        return body

    def emit_attrs(self, doc, obj):
        doc.emit_suggest_attr(obj['slug'])
        # Чтобы ноды типа conductor.maps_load_renderer_carparks находились
        # по "maps_load_renderer_carp" и по "renderer_carp" добавляем в значения саджестов
        # их части, начинающиеся с после разделителя: "maps_load_renderer_carparks",
        # "load_renderer_carparks" и т.д. Тогда запрос вида "load_renderer_carp*" будет
        # находить ноды
        for name_option in gen_options_by_special_chars(obj['slug']):
            if name_option != obj['slug']:
                doc.emit_suggest_attr(name_option)

        for suggest_element in obj['name'].values():
            doc.emit_suggest_attr(suggest_element)
            for part in get_suggest_parts(suggest_element):
                doc.emit_suggest_attr(part)

        if len(obj['aliases']) < ALIASES_MAX_COUNT_FOR_SUGGEST:
            for alias in obj['aliases']:
                for name in alias['name'].values():
                    doc.emit_suggest_attr(name)

        doc.emit_search_attr('s_slug', obj['slug'])
        doc.emit_search_attr('s_parent_path', obj['parent_path'])
        doc.emit_search_attr('s_system', obj['system']['slug'])
        if obj.get('updated_at'):
            doc.emit_search_attr('i_updated', date_to_timestamp(obj['updated_at']))

    def emit_factors(self, doc, obj):
        doc.emit_factor('isRolenode', 1)

    def create_snippet(self, obj, lang='ru', **kwargs):
        data = {
            'slug': obj['slug'],
            'system': obj['system']['slug'],
            'slug_path': obj['slug_path'],
            'parent_path': obj['parent_path'],
            'title': get_by_lang(obj['name'], lang),
            'help': get_by_lang(obj['help'], lang),
            'human_short': obj['human_short'],
            'object_type': self.source,
        }
        return RolenodeSnippet(data)
