from django.utils.dateparse import parse_datetime

from intranet.search.core.sources.idm.base import SourceBase
from intranet.search.core.sources.utils import (
    get_by_lang,
    get_document_url,
    get_suggest_parts,
    timestamp_to_utc_date,
    get_persons,
)
from intranet.search.core.snippets.idm import UserSnippet


class Source(SourceBase):
    """Индексатор нод для людей idm"""

    source = 'users'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

    def do_fetch(self, obj, **kwargs):
        obj['first_name'] = obj['name']['first']
        obj['last_name'] = obj['name']['last']
        obj['is_active'] = not obj['official']['is_dismissed']
        self.next('create', obj=obj)

    def fetch_objects(self, start_id=None):
        lookup = {'_sort': '-id', '_fields': 'id,login,name,official.is_dismissed,work_email,department_group,_meta'}
        query = []
        if start_id is not None:
            query.append(f'id < {start_id}')
        if self.options['ts']:
            created_filter = timestamp_to_utc_date(self.options['ts'])
            since_query = f'_meta.modified_at >= "{created_filter.isoformat()}"'
            query.append(since_query)
        if self.options['keys']:
            lookup['login'] = ','.join(self.options['keys'])
        if query:
            lookup['_query'] = ' and '.join(query)
        return get_persons(lookup=lookup)

    def get_doc_url(self, obj):
        return get_document_url('people').format(login=obj['login'])

    def get_doc_updated(self, obj):
        return parse_datetime(obj['_meta']['modified_at'])

    def create_body(self, obj, **kwargs):
        first_name = get_by_lang(obj['first_name'], 'ru')
        last_name = get_by_lang(obj['last_name'], 'ru')
        fullname = first_name + ' ' + last_name
        return {
            'slug': obj['login'],
            'name': fullname,
        }

    def emit_attrs(self, doc, obj):
        doc.emit_suggest_attr(obj['login'])
        for field in ('first_name', 'last_name'):
            for suggest_element in obj[field].values():
                doc.emit_suggest_attr(suggest_element)
                for part in get_suggest_parts(suggest_element):
                    doc.emit_suggest_attr(part)

        doc.emit_search_attr('i_is_active', int(obj['is_active']))
        doc.emit_search_attr('s_login', obj['login'])
        doc.emit_search_attr('s_type', 'user')

    def emit_factors(self, doc, obj):
        doc.emit_factor('userIsActive', int(obj['is_active']))
        doc.emit_factor('isUser', 1)

        # добавляем группировочные атрибуты для подсчета пользовательских факторов
        if obj.get('department_group'):
            doc.emit_group_attr('department', int(obj['department_group']['id']))
            if obj['department_group'].get('parent'):
                doc.emit_group_attr('parent_department', int(obj['department_group']['parent']['id']))

    def create_snippet(self, obj, lang='ru', **kwargs):
        data = {
            'login': obj['login'],
            'email': obj['work_email'],
            'url': self.get_doc_url(obj),
            'first_name': get_by_lang(obj['first_name'], lang),
            'last_name': get_by_lang(obj['last_name'], lang),
            'object_type': self.source,
            'is_active': obj['is_active'],
            'department_name': '',
        }
        if obj.get('department_group'):
            data['department_name'] = get_by_lang(obj['department_group']['department']['name']['full'], lang)
        return UserSnippet(data)
