from django.conf import settings
from intranet.search.core.sources.map.map import BaseOfficeIndexer
from intranet.search.core.snippets.map import EquipmentSnippet

from intranet.search.core.sources.map.utils import equipment_types_data


class Source(BaseOfficeIndexer):
    """
    Индексатор оборудования
    """

    def __init__(self, options, *args, **kwargs):
        self.equipment_url = settings.ISEARCH['map']['urls']['equipment']
        super().__init__(options, *args, staff_repo_name='equipment', **kwargs)

    @property
    def searched_type(self):
        return {'ru': 'оборудование', 'en': 'equipment'}

    def do_walk(self, **kwargs):
        equipment_items = self.office_stuff_repo.getiter(
            lookup={'type': ','.join(equipment_types_data.keys())}
        )

        for item in equipment_items:
            self.next('create', obj=item)

    def create_body(self, obj):
        equipment_type = obj['type']
        equipment_floor = obj['floor']
        equipment_office = equipment_floor['office']
        body = {
            'name': obj['name'],
            'name_dns': obj['hostname'],
            'description': obj['description'],
            'types': {
                'ru': equipment_types_data[equipment_type]['ru'],
                'en': equipment_types_data[equipment_type]['en'],
            },
            'place': {
                'floor': equipment_floor.get('name'),
                'office': equipment_office.get('name'),
            },
        }
        return body

    def emit_additional_attributes(self, doc, obj):
        if 'id' in obj['floor']:
            doc.emit_group_attr('floor', int(obj['floor']['id']))
            if 'id' in obj['floor']['office']:
                doc.emit_group_attr('office', int(obj['floor']['office']['id']))

    def emit_stat_factors(self, doc, obj):
        doc.emit_factor('isDeleted', int(obj['is_deleted']))

    def append_facets(self, doc, obj):
        super().append_facets(doc, obj)
        equipment_type = obj['type']
        current_equipment_type = equipment_types_data[equipment_type]
        doc.emit_facet_attr('equipment_type', f"equipment_type_{current_equipment_type['id']}",
                            current_equipment_type['ru'],
                            current_equipment_type['en'])

    def emit_final(self, doc, obj, names_are_required=True):
        super().emit_final(doc, obj, False)

    def create_snippet(self, obj, lang='ru'):
        equipment_type = obj['type']
        snippet = {
            'id': obj['id'],
            'name': obj['name'],
            'name_dns': obj['hostname'],
            'description': obj['description'],
            'coord_x': obj['location']['x'],
            'coord_y': obj['location']['y'],
            'floor_id': obj['floor']['id'],
            'type_id': equipment_types_data[equipment_type]['id'],
            'type_ru': equipment_types_data[equipment_type]['ru'],
            'type_en': equipment_types_data[equipment_type]['en'],
            'type': obj['type'],
        }

        self.add_geo_data(snippet, obj, lang)

        return EquipmentSnippet(snippet)

    def gen_url(self, obj):
        search_url = self.equipment_url.format(obj['id'])
        return search_url
