from intranet.search.core.utils import get_ids_repository
from intranet.search.core.swarm import Indexer
from intranet.search.core.sources.map.utils import split_by_not_letters

from intranet.search.core.sources.map.utils import rooms_types_data


class BaseOfficeIndexer(Indexer):
    """
    Базовый индексатор всего, что есть в офисах
    """

    def __init__(self, options, *args, **kwargs):
        super().__init__(options)
        repo_name = kwargs.pop('staff_repo_name')
        if not repo_name:
            raise TypeError('staff_repo_name parameter is required')
        self.office_stuff_repo = get_ids_repository('staff', repo_name)

    @property
    def searched_type(self):
        raise NotImplementedError('search_type function must return name of indexed office stuff type')

    def _prepare_lookup_room_type_filter(self, **kwargs):
        lookup_filters = dict()
        room_type_filter = set()

        if 'room_type' in kwargs:
            room_type_filter = kwargs.get('room_type')
            if isinstance(room_type_filter, str):
                room_type_filter = {room_type_filter}
            if isinstance(room_type_filter, list):
                room_type_filter = set(room_type_filter)

        if 'excluded_room_types' in kwargs:
            if len(room_type_filter) == 0:
                room_type_filter = set(rooms_types_data.keys())
            excluded_types = kwargs.get('excluded_room_types')
            if isinstance(excluded_types, str):
                excluded_types = [excluded_types]
            if isinstance(excluded_types, (list, set)):
                for excluded_type in excluded_types:
                    room_type_filter.discard(excluded_type)

        if len(room_type_filter) > 0:
            lookup_filters = {'type': ','.join(room_type_filter)}

        return lookup_filters

    def do_walk(self, **kwargs):
        lookup_filter = self._prepare_lookup_room_type_filter(**kwargs)
        rooms = self.office_stuff_repo.getiter(lookup=lookup_filter)
        for room in rooms:
            if room['is_deleted'] or room['floor']['office']['is_deleted']:
                self.do_delete(room)
            else:
                self.next('create', obj=room)

    def do_create(self, obj, **kwargs):
        search_url = self.gen_url(obj)
        doc = self.create_document(search_url)

        snippet = self.create_snippet(obj)
        doc.emit_snippet(snippet)

        self.emit_final(doc, obj)

        body = self.create_body(obj)
        doc.emit_body(body)

        self.next('store', document=doc)

    def base_body(self, obj):
        obj_type = obj['type']
        body = {
            'place': {
                'floor': obj['floor']['name'].get('ru'),
                'floor_en': obj['floor']['name'].get('en'),
                'office': obj['floor']['office']['name'],
            },
            'types': {
                'type_ru': rooms_types_data[obj_type],
                'type_en': obj_type,
            },
            'names': {
                'alternative': obj['name']['alternative'],
                'exchange': obj['name']['exchange'],
                'name': self.gen_names(obj['name']['display']),
            },
        }
        return body

    def create_body(self, obj):
        raise NotImplementedError()

    def emit_additional_attributes(self, doc, obj):
        raise NotImplementedError()

    def emit_names(self, doc, name):
        possible_names = self.gen_names(name)
        for emitted_name in possible_names:
            doc.emit_suggest_attr(emitted_name)
        splitted_name = split_by_not_letters(name)
        for name_part in splitted_name:
            if len(name_part) > 1:
                doc.emit_suggest_attr(name_part)

    def emit_stat_factors(self, doc, obj):
        pass

    def append_facets(self, doc, obj):
        doc.emit_facet_attr('country', f"country_id_{obj['floor']['office']['city']['country']['id']}",
                            obj['floor']['office']['city']['country']['name']['ru'],
                            obj['floor']['office']['city']['country']['name']['en']
                            )
        doc.emit_facet_attr('city', f"city_id_{obj['floor']['office']['city']['id']}",
                            obj['floor']['office']['city']['name']['ru'],
                            obj['floor']['office']['city']['name']['en']
                            )
        doc.emit_facet_attr('office', f"office_id_{obj['floor']['office']['id']}",
                            obj['floor']['office']['name']['ru'],
                            obj['floor']['office']['name']['ru']
                            )
        doc.emit_facet_attr('floor', f"floor_id_{obj['floor']['id']}",
                            obj['floor']['name']['ru'],
                            obj['floor']['name']['en'])
        doc.emit_facet_attr('what', f"target_object_{self.searched_type['ru']}",
                            self.searched_type['ru'],
                            self.searched_type['en']
                            )

    def emit_final(self, doc, obj, names_are_required=True):
        self.emit_stat_factors(doc, obj)
        self.emit_additional_attributes(doc, obj)
        self.append_facets(doc, obj)
        if names_are_required:
            self.emit_names(doc, obj['name']['display'])

    def do_delete(self, obj, **kwargs):
        url = self.gen_url(obj)
        doc = self.create_document(url)
        self.next('store', document=doc, delete=True)

    def base_snippet(self, obj, lang='ru'):
        obj_type = obj['type']
        snippet = {
            'id': obj['id'],
            'name': obj['name']['display'],
            'map_url': self.gen_map_url(obj),
            'floor_id': obj['floor']['id'],
            'type_ru': rooms_types_data[obj_type],
            'type_en': obj_type,
        }
        self.add_geo_data(snippet, obj, lang)
        return snippet

    def add_geo_data(self, snippet, obj, lang):
        # по структуре словаря obj: https://staff-api.yandex-team.ru/v3/rooms?_doc=1
        if 'name' in obj['floor']:
            snippet['floor_name'] = obj['floor']['name'].get(lang)
        if 'id' in obj['floor']['office']:
            snippet['office_id'] = obj['floor']['office']['id']
            snippet['office_name'] = obj['floor']['office']['name'][lang]
        if obj['floor']['office']['city']:
            snippet['city_id'] = obj['floor']['office']['city']['id']
            snippet['city_name'] = obj['floor']['office']['city']['name'][lang]

    def create_snippet(self, obj, lang='ru'):
        raise NotImplementedError()

    def gen_map_url(self, obj):
        raise NotImplementedError()

    def gen_url(self, obj):
        raise NotImplementedError()

    def gen_names(self, name):
        return [name]


class BaseRoomIndexer(BaseOfficeIndexer):
    def append_facets(self, doc, obj):
        super().append_facets(doc, obj)
        room_type = obj['type']
        doc.emit_facet_attr('room_type', f"room_type_{room_type}",
                            rooms_types_data[room_type],
                            room_type.capitalize())

    def emit_stat_factors(self, doc, obj):
        equipment = obj['equipment']
        utilities = [
            equipment['cork_board'], equipment['desk'], bool(equipment['game_console']),
            equipment['guest_wifi'], equipment['marker_board'], equipment['projector'],
            equipment['screen'], equipment['seats'], bool(equipment['video_conferencing']),
            equipment['voice_conferencing']
        ]
        cork_board, desk, game, wifi, marker_board, projector, screen, seats, video, voice = utilities
        # sometimes we get 'capacity': ''
        doc.emit_factor('roomCapacity', int(bool(obj.get('capacity'))))
        doc.emit_factor('hasCorkBoard', int(cork_board))
        doc.emit_factor('hasDesk', int(desk))
        doc.emit_factor('hasGameConsole', int(game))
        doc.emit_factor('hasGuestWifi', int(wifi))
        doc.emit_factor('hasMarkerBoard', int(marker_board))
        doc.emit_factor('projectorCount', int(projector))
        doc.emit_factor('screenCount', int(screen))
        doc.emit_factor('seatsCount', int(seats))
        doc.emit_factor('hasVideoConferencing', int(video))
        doc.emit_factor('hasVoiceConferencing', int(voice))
        doc.emit_factor('isBookable', int(obj['is_bookable']))
        doc.emit_factor('isCabin', int(obj['is_cabin']))
        doc.emit_factor('isDeleted', int(obj['is_deleted']))
