from collections.abc import Iterable
from django.utils.dateparse import parse_datetime

from intranet.search.core.snippets.moebius import MoebiusSnippet
from intranet.search.core.sources.moebius.client import MoebiusApiClient
from intranet.search.core.swarm.api_indexer import PagedApiIndexer
from intranet.search.core.sources.utils import date_as_factor


class Source(PagedApiIndexer):
    snippet_languages = ('ru',)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.api_client = MoebiusApiClient()

    def create_snippet(self, course, lang='ru', **kwargs):
        cover = None
        if course['cover']:
            cover = {
                'id': course['cover'].get('id'),
                'url': course['cover'].get('file'),
                'type': course['cover'].get('type'),
                'nda': course['cover'].get('nda'),
            }
        snippet = {
            'url': course['url'],
            'title': course['name'][lang],
            'description': course['content'][lang][:300],
            'subject': course['subject'][lang][:300],
            'cover': cover,
            'rating': float(course['average_score']),
        }
        return MoebiusSnippet(snippet)

    def fetch_objects(self, page: int = 1) -> Iterable:
        response = self.api_client.fetch_page(page)
        return response['results']

    def get_counts(self):
        data = self.api_client.fetch_page()
        docs_count = data['count']
        # warning: moebius has their internal limit for the `limit` field, so we cannot trust self.api_client.page_size
        per_page = len(data['results'])
        if per_page == 0:
            return 0, docs_count
        pages_count = (docs_count + per_page - 1) // per_page
        return pages_count, docs_count

    def get_doc_url(self, course):
        return course['url']

    def get_doc_updated(self, course):
        return parse_datetime(course['updated'])

    def get_doc_created(self, course):
        return parse_datetime(course['created'])

    def emit_factors(self, doc, obj):
        doc.emit_factor('updated', date_as_factor(self.get_doc_updated(obj)))
        doc.emit_factor('created', date_as_factor(self.get_doc_created(obj)))
        doc.emit_factor('hasCover', int(bool(obj.get('cover'))))
        doc.emit_factor('rating', float(obj['average_score']))

    def create_body(self, course, **kwargs):
        body = {
            'description': course['content']['ru'][:300],
            'z_ns_hidden': {
                'title': course['name']['ru'],
                'subject': course['subject'].get('ru'),
                'rating': course['average_score'],
            },
        }
        return body
