from logging import getLogger

from intranet.search.core.utils import get_ids_repository

from intranet.search.core.sources.utils import get_document_url, get_by_lang, get_suggest_parts, Services
from intranet.search.core.snippets.people import GroupSnippet
from intranet.search.core.swarm.api_indexer import PagedApiIndexer


log = getLogger(__name__)


class Source(PagedApiIndexer):
    """ Индексатор групп стаффа
    """

    def __init__(self, options):
        super().__init__(options)
        self.groups_repo = get_ids_repository('staff', 'group', timeout=5)
        self.services = Services(self.cache_storage, fields=('id', 'name', 'slug'))

    def fetch_objects(self):
        params = {
            '_fields': 'id,name,url,service,department.id,department.url,department.name,'
                       'type,is_deleted,description,affiliation_counters',
            '_limit': 100,
            'is_deleted': 'false,true',
        }
        if self.options['keys']:
            params['url'] = ','.join(self.options['keys'])
        return self.groups_repo.getiter(params)

    def do_walk(self, **kwargs):
        for obj in self.fetch_objects():
            if obj['is_deleted']:
                self.next('store', delete=True, document=self.create_document(self.get_doc_url(obj)))
            else:
                self.next('fetch', obj=obj)

    def do_fetch(self, obj, **kwargs):
        if obj['service'] and obj['service']['id']:
            try:
                obj['service'] = self.services[obj['service']['id']]
            except Exception as e:
                log.error(e)
                obj['service'] = {}
        if not obj['service'].get('name'):
            obj['service'] = {}

        if obj['department'] and not obj['department']['id']:
            obj['department'] = {}

        self.next('create', obj=obj)

    def get_doc_url(self, obj):
        return get_document_url('groups').format(slug=obj['url'])

    def create_body(self, obj):
        body = {
            'group_id': obj['id'],
            'group_name': [obj['name']],
            'group_slug': obj['url'],
            'group_description': obj['description'],
        }
        if obj['department']:
            body['group_name'].extend(obj['department']['name']['full'].values())
        if obj['service']:
            body['group_name'].extend(obj['service']['name'].values())
        body['group_name'] = list(set(body['group_name']))
        return body

    def create_snippet(self, obj, lang='ru'):
        data = {
            'id': obj['id'],
            'title': obj['name'],
            'slug': obj['url'],
            'type': obj['type'],
            'url': self.get_doc_url(obj),
            'affiliation_counters': obj['affiliation_counters'],
        }
        if obj['service']:
            data['service'] = {
                'id': obj['service']['id'],
                'title': get_by_lang(obj['service'].get('name', {}), lang),
                'slug': obj['service'].get('slug')
            }
            if obj['service'].get('name', {}).get('ru', '') == obj['name']:
                data['title'] = get_by_lang(obj['service'].get('name', {}), lang)

        if obj['department']:
            data['department'] = {
                'id': obj['department']['id'],
                'title': get_by_lang(obj['department']['name']['full'], lang),
                'slug': obj['department']['url']
            }
            if obj['department']['name']['full']['ru'] == obj['name']:
                data['title'] = get_by_lang(obj['department']['name']['full'], lang)
        return GroupSnippet(data)

    def emit_factors(self, doc, obj):
        doc.emit_factor('is%s' % obj['type'].title(), 1)

    def emit_attrs(self, doc, obj):
        doc.emit_search_attr('s_type', obj['type'])
        doc.emit_search_attr('s_url', obj['url'])
        doc.emit_search_attr('i_group_id', obj['id'])
        doc.emit_suggest_attr(obj['url'])
        doc.emit_suggest_attr(str(obj['id']))

        names = [obj['name']]
        if obj['department']:
            names.extend(obj['department']['name']['full'].values())
            names.extend(obj['department']['name']['short'].values())
            names.append(obj['department'].get('url'))
        if obj['service'] and obj['service'].get('name'):
            names.extend(obj['service']['name'].values())
            names.append(obj['service'].get('slug'))

        for name in set(names):
            if not name:
                continue
            doc.emit_suggest_attr(name)
            for part in get_suggest_parts(name):
                doc.emit_suggest_attr(part)
