import datetime
import logging

import phonenumbers
import pytils

from intranet.search.core.snippets.people import PeopleSnippet
from intranet.search.core.sources.people.utils import flatten_groups
from intranet.search.core.sources.utils import (
    get_by_lang,
    get_persons,
    get_person_details,
    get_person_fio,
    get_suggest_parts,
    hash_factor,
    Metrix,
    normalize,
    normalize_email,
    StaffGroups,
    timestamp_to_utc_date,
)
from intranet.search.core.swarm.api_indexer import Indexer
from intranet.search.core.utils import get_ids_repository, reraise_as_recoverable, http

from . import utils


log = logging.getLogger(__name__)

FORMER_DEPARTMENT = 'Бывшие сотрудники'
FORMER_DEPARTMENT_EN = 'Former'
FORMER_DEPARTMENT_URL = 'former'

HOMEWORKER_OFFICE_ID = 14
VIRTUAL_OFFICE_ID = 144
ASSESSOR_DEPARTMENT_ID = 1262


dep_repo = get_ids_repository('staff', 'group')


class Source(Indexer):
    snippet_languages = ('ru', 'en')

    def __init__(self, options):
        super().__init__(options)
        self.services = utils.PeopleServiceRoles(self.cache_storage)
        self.metrix = Metrix(
            cache=self.cache_storage,
            source='people',
            regex=r'^/(?P<key>[\w\-]+)/?$',
        )
        self.departments = StaffGroups(self.cache_storage,
                                       key_field='department.id', fields=('department', ))

    def do_setup(self, **kwargs):
        try:
            if not self.is_delta and not self.options['keys']:
                lookup = {'service__state__in': 'develop,supported', 'page_size': 500}
                self.services.prepare_services(lookup)
        except Exception:
            log.exception('Cannot update services in people indexation')
        try:
            if not self.is_delta and not self.options['keys']:
                self.metrix.prepare_metrix()
        except Exception:
            log.exception('Cannot prepare metrix data')
        super().do_setup()

    @reraise_as_recoverable(*http.ERRORS)
    def do_walk(self, **kwargs):
        count = 0
        last_id = None
        has_objects = True
        while has_objects:
            objects = self.fetch_objects(last_id)
            has_objects = bool(objects)
            count += len(objects)
            for obj in objects:
                last_id = obj['id']
                self.next('fetch', obj=obj)
        log.info("Walk fetch %s objects", count)

    def fetch_objects(self, start_id=None):
        lookup = {'_sort': '-id', '_fields': 'id,login'}
        query = []
        if start_id is not None:
            query.append(f'id < {start_id}')
        if self.options['ts']:
            created_filter = timestamp_to_utc_date(self.options['ts'])
            since_query = f'_meta.modified_at >= "{created_filter.isoformat()}"'
            query.append(since_query)
        if self.options['keys']:
            lookup['login'] = ','.join(self.options['keys'])
        if query:
            lookup['_query'] = ' and '.join(query)
        return get_persons(lookup=lookup)

    def do_fetch(self, obj=None, **kwargs):
        person_data = get_person_details(obj['login'])

        head_deps_query = {'department.heads.person.login': obj['login'],
                           'type': 'department',
                           '_fields': 'department',
                           '_limit': 500}
        head_deps = dep_repo.get(lookup=head_deps_query, timeout=5)
        if self.is_delta:
            person_data['abc_data'] = self.services[obj['login']] or {}
        else:
            # если это не дельта, берем данные только из подготовленного кеша,
            # а не пытаемся их фетчить, если кеш пуст
            person_data['abc_data'] = self.services.get(obj['login'], {})

        self.next('create', data=person_data, head_deps_data=head_deps)

    def get_person_direction(self, dep_group):
        # Сюда мы попадаем только если апи стаффа уже новое и для депарментов есть kind

        def _is_direction(dp):
            kind = dp['department'].get('kind')
            return kind and kind['slug'] == 'direction'

        # Не работает ли человек непосредственно в корне направления?
        if _is_direction(dep_group):
            return dep_group

        # Если нет, то среди предков департамента ищем направление
        for dep in dep_group['ancestors']:
            if _is_direction(dep):
                return dep

        # Сюда попадаем, если все совсем плохо
        return {}

    def define_person_direction(self, data):
        direction_info = {}

        if 'kind' in data['department_group']['department']:
            direction = self.get_person_direction(data['department_group'])
        else:
            try:
                if data['official']['affiliation'] == 'external':
                    direction = data['department_group']['ancestors'][0]
                else:
                    direction = data['department_group']['ancestors'][1]
            except IndexError:
                direction = data['department_group']

        if 'department' in direction and 'name' in direction['department']:
            direction_info = {
                'id': direction['department']['id'],
                'name': direction['department']['name'],
                'url': direction.get('url'),
            }
        elif 'id' in direction and 'name' in direction:
            direction_info = {
                'id': direction['id'],
                'name': {
                    'full': {
                        'ru': direction['name'],
                        'en': '',
                    },
                    'short': {
                        'ru': direction['name'],
                        'en': '',
                    }
                },
                'url': direction.get('url'),
            }

        return direction_info

    def do_create(self, data, head_deps_data, **kwargs):
        doc_url = 'https://staff.yandex-team.ru/%s' % data['login']
        doc = self.create_document(doc_url)

        direction = self.define_person_direction(data)

        doc.emit_search_attr('s_login', data['login'])
        doc.emit_search_attr('i_is_person', 1)
        doc.emit_search_attr('i_is_service', 0)
        doc.emit_search_attr('i_is_department', 0)

        doc.emit_search_attr('i_is_dismissed', int(data['official']['is_dismissed']))
        doc.emit_search_attr('s_affiliation', data['official']['affiliation'])
        doc.emit_search_attr('i_is_memorial', int(bool(data.get('memorial'))))
        doc.emit_search_attr('i_is_robot', int(data['official']['is_robot']))

        for key in data['keys']:
            try:
                # в качестве разделителя для атрибута используем пробел
                fingerprint_1 = utils.get_fingerprint(key['key'], delim=' ')
                fingerprint_2 = utils.get_fingerprint(key['key'], delim=':')
            except Exception:
                continue
            doc.emit_search_attr('s_fingerprint', fingerprint_1)
            doc.emit_search_attr('s_fingerprint', fingerprint_2)

        if 'id' in direction:
            doc.emit_search_attr('s_direction_ru', direction['name']['short']['ru'])
            doc.emit_search_attr('s_direction_en', direction['name']['short']['en'])

        doc.emit_suggest_attr(data['login'])

        for lang in ('en', 'ru'):
            first_name = get_by_lang(data['name']['first'], lang)
            last_name = get_by_lang(data['name']['last'], lang)
            for name in [first_name, last_name] + list(get_suggest_parts(last_name)):
                doc.emit_suggest_attr(name)

        if data['work_phone']:
            doc.emit_suggest_attr(str(data['work_phone']))

        for car in data.get('cars', []):
            if car['plate']:
                for car_number in utils.get_car_nums(car['plate']):
                    doc.emit_suggest_attr(car_number)

        for phone in data.get('phones', []):
            if phone['type'] == 'mobile':
                for phone_number in utils.gen_phones(phone['number']):
                    doc.emit_suggest_attr(phone_number)

        body = self.create_body(data, direction)
        doc.emit_body(body)

        self.append_factors(doc, data, head_deps_data)

        for lang in ('ru', 'en'):
            snippet = self.create_snippet(data, direction, lang)
            doc.emit_snippet(snippet, lang)

        doc.emit_group_attr('staff_id', data['id'])
        for group in flatten_groups(data.get('groups', [])):
            doc.emit_search_attr('i_group_id', group['id'])
            if group.get('type') in ('department', 'service'):
                doc.emit_group_attr(group['type'], group['id'])

        self.append_facets(data, direction, doc)

        self.next('store', document=doc, body_format='json')

    def append_facets(self, data, direction, doc):
        for service in data['abc_data'].get('services', {}).values():
            doc.emit_facet_attr('service', service['service']['id'] or 'none',
                                service['service']['name']['ru'],
                                service['service']['name']['en'])

        location = utils.get_location(data)
        city = location['office'].get('city', {})
        if city:
            doc.emit_facet_attr('city', city['id'] or 'none', city['name']['ru'], city['name']['en'])
        else:
            doc.emit_facet_attr('city', 'none', 'Нет города', 'No city')

        if 'id' in direction:
            doc.emit_facet_attr('direction', direction['id'],
                                 direction['name']['short']['ru'] or direction['name']['full']['ru'],
                                 direction['name']['short']['en'] or direction['name']['full']['en'])
        else:
            doc.emit_facet_attr('direction', 'none', 'Нет направления', 'No direction')

        if data['official']['is_robot']:
            doc.emit_facet_attr('type', 'robot', 'Робот', 'Robot')
        else:
            doc.emit_facet_attr('type', 'person', 'Человек', 'Person')

    def create_body(self, data, direction):
        contacts = {}
        for contact in data.get('accounts', []):
            c_type = contact['type']
            if c_type in contacts:
                contacts[c_type].append(contact['value'])
            else:
                contacts[c_type] = utils.get_contact_type_forms(c_type) + [contact['value']]

        work_email = ([data['work_email']] +
                      utils.contact_type['work_email'] +
                      list(utils.contact_labels['work_email'].values()))

        work_email += [m['address'] for m in data['emails']]

        body = {
            'car': {
                '!car_model': [car['model'] for car in data.get('cars', {})],
                '!car_number': [utils.get_car_nums(car['plate']) for car in data.get('cars', {})],
            },
            '!bicycle': {
                '!bicycle_description': [b['description'] for b in data.get('bicycles', [])],
                '!bicycle_number': [b['plate'] for b in data.get('bicycles', [])],
                'type': utils.bicycle_types,
            },
            'contacts': {
                'work_email': work_email,
                'moi_krug': contacts.get('moi_krug'),
                'jabber': contacts.get('jabber'),
                'login_passport': data['yandex']['login'],
                'address': data['personal']['address'],
            },
            'phones': {
                'work_phone': {
                    'number': data['work_phone'],
                    'type': utils.phone_type['work'],
                    },
                'personal': [{
                    'description': p['description'],
                    'number': list(utils.gen_phones(p['number'])),
                    'type': utils.phone_type[p['type']],
                } for p in data.get('phones', [])],
            },
            'name': {
                'fio_ru': get_person_fio(data, 'ru'),
                'fio_en': get_person_fio(data, 'en'),
                'login': {
                    'original': data['login'],
                    'translated': pytils.translit.detranslify(data['login']),
                }
            },
            'work': {
                'department': data['department_group']['department'].get('name', {}),
                'department_url': data['department_group']['department'].get('url'),
                'position': data['official']['position'],
            },
            'personal': {
                'sex': utils.get_gender_forms(data['personal']['gender']),
                'mobile_phone_model': data['personal']['mobile_phone_model'],
            },
            'direction': direction.get('name', {}),
            'affiliation': data['official']['affiliation'],
        }

        if data['official']['is_dismissed']:
            additional = self.create_dismissed_body(data)
        else:
            additional = self.create_additional_body(data, contacts)

        for key, value in additional.items():
            if key in body and isinstance(value, dict):
                body[key].update(value)
            else:
                body[key] = value

        return body

    def create_dismissed_body(self, data):
        body = {
            'work': {
                'ex_dep': {
                    'department_ru': FORMER_DEPARTMENT,
                    'department_en': FORMER_DEPARTMENT_EN,
                    'department_url': FORMER_DEPARTMENT_URL,
                },
            }
        }
        return body

    def create_additional_body(self, data, contacts):
        services = data['abc_data'].get('services', {}).values()
        # может быть более одной роли у человека в сервисе
        roles = []
        for service in services:
            roles.append([(r['name']['ru'], r['name']['en']) for r in service.get('roles', [])])

        if data['personal']['birthday']:
            birthday = datetime.datetime.strptime(data['personal']['birthday'],
                                                  '%Y-%m-%d')
            age = '%s лет' % ((datetime.datetime.now() - birthday).days / 365)
        else:
            age = ''

        location = utils.get_location(data)

        group_info = self.departments[data['department_group']['department'].get('id')]
        dep_info = group_info.get('department') or {}

        # Собираем из всех сервисов пользователей зону: <должность> <сервис>.
        # Например: "Разработчик Staff", "Администратор Фемида", "Руководитель сервиса Фемида"
        service_roles = []
        for service in services:
            for lang in self.snippet_languages:
                for role in service['roles']:
                    service_role = ' '.join([
                        get_by_lang(role['name'], lang),
                        get_by_lang(service['service']['name'], lang),
                    ])
                    service_roles.append(service_role)

        body = {
            'contacts': contacts,
            'work': {
                'department_wiki': dep_info.get('contacts', {}).get('wiki'),
                'department_ml':  dep_info.get('contacts', {}).get('maillists'),
                'department_description': dep_info.get('description'),
                'deps': [dep['department'].get('name')
                         for dep in data['department_group']['ancestors']],
                'duties': data['official']['duties'],
                'roles': roles,
            },
            'personal': {
                'education': {
                    'status': utils.get_edu('status', data['education']['status']),
                    'area': utils.get_edu('area', data['education']['area']),
                    'place': data['education']['place'],
                },
                'tshirt': data['personal']['tshirt_size'],
                'birth_date': data['personal']['birthday'],
                'age': age,
                'family_status': utils.get_family_status(data['personal']),
                'about': data['personal']['about'],
            },
            'place': {
                'floor': {
                    'floor_name': location['floor'].get('name'),
                    'number': ('%s этаж' % location['floor']['number']
                                    if location['floor'].get('number') else ''),
                },
                'office': {
                    'office_name': location['office'].get('name', ''),
                    'city': location['office'].get('city', {}).get('name', ''),
                },
                'description': location['description'],
            },
            'person_service_name': service_roles,
        }

        additional_contacts = {
            'login_passport': data['yandex']['login'],
            'address': data['personal']['address'],
        }

        body['contacts'].update(additional_contacts)

        return body

    def append_factors(self, doc, data, head_deps_data):
        profile = self.profile_storage.get('person', data['login'])

        doc.emit_factor('urlHash', hash_factor(doc.url))
        doc.emit_factor('wikiLinksCount', normalize(profile.get('wikiLinksCount', 0), 500))

        is_former = (data['official']['is_dismissed'] and data.get('memorial') is None)
        doc.emit_factor('isFormer', int(is_former))

        if data['official']['affiliation'] == 'external':
            doc.emit_factor('isOutstaff', 1)

        doc.emit_factor('isVolozh', int(data['login'] == 'volozh'))

        person_head_roles = []

        if head_deps_data:
            for dep in head_deps_data:
                res = [
                    p['role'] for p in dep['department'].get('heads', [])
                    if p['person']['login'] == data['login']
                ]
                person_head_roles += res

        if 'chief' in person_head_roles:
            doc.emit_factor('isHead', 1)
        elif 'deputy' in person_head_roles:
            doc.emit_factor('isDeputyHead', 1)

        if data['official']['is_homeworker']:
            doc.emit_factor('isHomeworker', 1)

        if data['location']['office'].get('id') == VIRTUAL_OFFICE_ID:
            doc.emit_factor('isInVirtualOffice', 1)

        dep_ids = [dep['id'] for dep in data['department_group'].get('ancestors', [])]

        if (ASSESSOR_DEPARTMENT_ID in dep_ids or
                data['official']['position'].get('en', '').lower() == 'assessor' or
                data['official']['position'].get('ru') == 'асессор'):
            doc.emit_factor('isAssessor', 1)

        if data['abc_data'].get('is_owner', False):
            doc.emit_factor('isTeamLead', 1)

        level = data['department_group']['department'].get('level', 10)
        doc.emit_factor('personDepartmentLevel', (10 - level)/10.)

        page_views = self.metrix[data['login']] or 0
        doc.emit_factor('pageViews', page_views / 600. if page_views < 600 else 1)

    def fill_teams(self, data, lang):
        for service in data['abc_data'].get('services', {}).values():
            for role in service['roles']:
                team = {
                    'service': {
                        'id': service['service']['id'],
                        'name': get_by_lang(service['service']['name'], lang),
                    },
                    'role': get_by_lang(role['name'], lang),
                }
                yield team

    def fill_services(self, data, lang):
        for service in data['abc_data'].get('services', {}).values():
            yield {
                'id': service['service']['id'],
                'name': get_by_lang(service['service']['name'], lang),
            }

    def fill_phones(self, data):
        if data.get('work_phone'):
            yield {'type': 'work',
                   '_type_forms': utils.phone_type['work'],
                   'value': str(data['work_phone']),
                   'description': '',
                   '_forms': [str(data['work_phone'])]}

        for phone in data.get('phones', []):
            try:
                parsed_number = phonenumbers.parse(phone['number'])
                number = phonenumbers.format_number(
                    parsed_number, phonenumbers.PhoneNumberFormat.INTERNATIONAL)
            except phonenumbers.NumberParseException:
                number = phone['number']
            yield {
                'type': phone['type'],
                '_type_forms': utils.phone_type[phone['type']],
                'description': phone['description'],
                'value': number,
                '_forms': list(utils.gen_phones(number)),
            }

    def fill_contacts(self, data, lang):
        for c in data.get('accounts', []):
            if c['value']:
                yield {
                    'type': c['type'],
                    'value': c['value'],
                    'label': utils.contact_labels.get(c['type'], {}).get(lang, c['type']),
                    '_type_forms': utils.get_contact_type_forms(c['type']),
                }

        if data['work_email']:
            yield {
                'type': 'work_email',
                'value': data['work_email'],
                'label': utils.contact_labels['work_email'][lang],
                '_type_forms': utils.get_contact_type_forms('work_email'),
            }

    def create_snippet(self, data, direction, language):
        dep = data['department_group']
        dep_info = self.departments[dep['department']['id']].get('department') or {}

        if 'name' in dep['department']:
            dep_name_short = get_by_lang(dep['department']['name'].get('short'), language)
            dep_name = get_by_lang(dep['department']['name'].get('full'), language)
        else:
            dep_name = dep_name_short = ''

        if direction:
            direction_info = self.departments[direction['id']].get('department') or {}

            direction_data = {
                'id': direction['id'],
                'url': direction['url'],
                'description': get_by_lang(direction_info.get('description'), language),
                'maillists': [normalize_email(m) for m in direction_info.get('contacts', {}).get('maillists') or []],
                'wiki': direction_info.get('contacts', {}).get('wiki') or '',
                'dep_name': get_by_lang(direction['name']['full'], language),
                'dep_name_short': get_by_lang(direction['name']['short'], language),
            }
        else:
            direction_data = {}

        snippet = {
            'is_person': True,
            'login': data['login'],
            'id': data['id'],  # id на стаффе
            'uid': data['uid'],  # яндексовый uid
            'is_dismissed': data['official']['is_dismissed'],
            'affiliation': data['official']['affiliation'],
            'is_robot': data['official']['is_robot'],
            'is_memorial': bool(data.get('memorial', False)),
            'department': {
                'id': dep['department']['id'],
                'url': dep.get('url'),
                'description': get_by_lang(dep_info.get('description'), language),
                'maillists': [normalize_email(m) for m in dep_info.get('maillists') or []],
                'wiki': dep_info.get('wiki') or '',
                'dep_name': dep_name,
                'dep_name_short': dep_name_short,
            },
            'direction': direction_data,
            'position': get_by_lang(data['official']['position'], language),
            'work_email': data.get('work_email', '')
        }

        need_to_hide_dismissed = (
            data['official']['is_dismissed']
            and not data['official'].get('staff_agreement', False)
            and not data.get('memorial', False)
        )
        if need_to_hide_dismissed:
            snippet.update(self.create_dismissed_snippet(data, language))
        else:
            snippet.update(self.create_additional_snippet(data, language))

        return PeopleSnippet(snippet, strict=False)

    def create_dismissed_snippet(self, data, language):
        data['replace_name'] = {
            'first': {
                'en': 'Former',
                'ru': 'Бывший'
            },
            'second': {
                'en': 'employee',
                'ru': 'сотрудник'
            }
        }
        return {
            'name': {
                'first': get_by_lang(data['replace_name']['first'], language),
                'last': get_by_lang(data['replace_name']['second'], language)
            }
        }

    def create_additional_snippet(self, data, language):
        location = utils.get_location(data)

        keys = []
        for key in data['keys']:
            try:
                fingerprint = utils.get_fingerprint(key['key'])
            except:
                continue
            keys.append({'description': key['description'], 'fingerprint': fingerprint})
        phones = list(self.fill_phones(data))
        contacts = list(self.fill_contacts(data, language))

        return {
            'name': {
                'first': get_by_lang(data['name']['first'], language),
                'last': get_by_lang(data['name']['last'], language),
                'middle': data['name'].get('middle', ''),
            },
            'duties': get_by_lang(data['official']['duties'], language),
            'phones': phones,
            'contacts': contacts,
            'work_email': data.get('work_email', ''),
            'cars': [{'model': c['model'],
                      'plate': c['plate'],
                      'number': utils.get_car_nums(c['plate'])}
                     for c in data.get('cars', [])],
            'bicycles': [{'description': b['description'], 'number': str(b['plate'])}
                         for b in data.get('bicycles', [])],
            'work_place': {
                'office': {
                    'name': get_by_lang(location['office'].get('name'), language),
                    'id': location['office'].get('id'),
                    'code': location['office'].get('code'),
                },
                'floor': {
                    'name': get_by_lang(location['floor'].get('name'), language),
                    'id': location['floor'].get('id'),
                },
                'table': {
                    'id': location['table'].get('id'),
                    'number': location['table'].get('number'),
                },
                'description': get_by_lang(location['description'], language),
            },
            'education': {
                'place': get_by_lang(data['education'].get('place'), language),
                'status': utils.get_edu('status', data['education']['status'])[language],
                'area': utils.get_edu('area', data['education']['area'])[language],
            },
            'services': list(self.fill_services(data, language)),
            'mobile_phone_model': data['personal']['mobile_phone_model'],
            'birthday': data['personal']['birthday'],
            'address': get_by_lang(data['personal']['address'], language),
            'tshirt_size': data['personal']['tshirt_size'],
            'yandex_login': data['yandex']['login'],
            'keys': keys,
        }
