import base64
import hashlib
import logging
import re
from collections import defaultdict
from copy import copy

import phonenumbers

from intranet.search.core.utils import get_ids_repository
from intranet.search.core.utils.cache import CacheAdapter

log = logging.getLogger(__name__)

bicycle_types = ['велосипед', 'вел']
phone = ['телефон', 'номер', 'позвонить', 'phone']
phone_type = {
    'mobile': ['мобильный', 'mobile'] + phone,
    'home': ['домашний', 'дом', 'home'] + phone,
    'work': ['рабочий', 'служебный'] + phone,
}

contact_type = {
    'personal_site': [],
    'site': [],
    'icq': ['аська'],
    'moi_krug': ['mk'],
    'twitter': [],
    'jabber': ['жаббер', 'джабер'],
    'personal_email': ['private email', 'e-mail', 'емейл', 'почта', 'домашняя почта', 'домашний емейл'],
    'work_email': ['email', 'e-mail', 'емейл', 'почта', 'рабочий емейл', 'рабочая почта'],
    'email': ['email', 'e-mail', 'емейл', 'почта'],
    'phone': phone,
    'skype': ['скайп'],
    'livejournal': ['livejournal', 'жж', 'живой журнал'],
    'another_work_email': [],
    'github': ['github', 'гитхаб'],
    'facebook': [],
    'vkontakte': [],
    'habrahabr': [],
    'instagram': [],
    'flickr': [],
    'tumblr': [],
    'blogspot': [],
    'telegram': [],
    'yamb': [],
    'staff': [],
}

contact_labels = {
    'personal_site': {'en': 'site', 'ru': 'сайт'},
    'site': {'en': 'site', 'ru': 'сайт'},
    'icq': {'en': 'icq', 'ru': 'icq'},
    'moi_krug': {'en': 'moikrug', 'ru': 'мой круг'},
    'twitter': {'en': 'twitter', 'ru': 'твиттер'},
    'jabber': {'en': 'jabber', 'ru': 'джаббер'},
    'personal_email': {'en': 'private email', 'ru': 'почта дома'},
    'work_email': {'en': 'email', 'ru': 'почта'},
    'skype': {'en': 'skype', 'ru': 'skype'},
    'livejournal': {'en': 'livejournal', 'ru': 'живой журнал'},
    'another_work_email': {'en': 'another work email', 'ru': 'ещё рабочая почта'},
    'github': {'en': 'github', 'ru': 'гитхаб'},
    'facebook': {'en': 'facebook', 'ru': 'facebook'},
    'vkontakte': {'en': 'vkontakte', 'ru': 'вконтакте'},
    'habrahabr': {'en': 'habrahabr', 'ru': 'хабрахабр'},
    'instagram': {'en': 'instagram', 'ru': 'instagram'},
    'flickr': {'en': 'flickr', 'ru': 'flickr'},
    'tumblr': {'en': 'tumblr', 'ru': 'tumblr'},
    'blogspot': {'en': 'blogspot', 'ru': 'blogspot'},
    'telegram': {'en': 'telegram', 'ru': 'telegram'},
    'yamb': {'en': 'yamb', 'ru': 'ямб'},
    'phone': {'en': 'phone', 'ru': 'телефон'}
}

education_area = {
    'technical': {
        'ru': 'техническое',
        'en': 'technical',
        '_forms': ['technical'],
    },
    'liberal': {
        'en': 'liberal',
        'ru': 'гуманитарное',
        '_forms': ['гуманитарное'],
    },
    'natural': {
        'en': 'natural',
        'ru': 'естественно-научное',
        '_forms': [],
        },
    'economic': {
        'en': 'economic',
        'ru': 'экономическое',
        '_forms': []
    },
    'business': {
        'en': 'business',
        'ru': 'деловое администрирование',
        '_forms': ['mba']
    },
}

education_status = {
    'specialist': {
        'ru': 'специалист',
        'en': 'specialist',
        '_forms': ['специалитет'],
    },
    'bachelor': {
        'ru': 'бакалавр',
        'en': 'bachelor',
        '_forms': ['бакалавриат']
    },
    'master': {
        'ru': 'магистр',
        'en': 'master',
        '_forms': ['магистратура'],
    },
    'secondary': {
        'ru': 'среднее',
        'en': 'secondary',
        '_forms': [],
    },
    'incomplete': {
        'ru': 'незаконченное высшее',
        'en': 'incomplete',
        '_forms': [],
    },
    'academic': {
        'ru': 'ученая степень',
        'en': 'academic',
        '_forms': ['кандидат', 'доктор', 'phd', 'Ph.D.'],
    },
}

GENDER_FORMS = {
    'male': ['мужчина', 'мальчик', 'парень', 'male', 'кун'],
    'female': ['женщина', 'девочка', 'девушка', 'female', 'тян'],
}


class PeopleServiceRoles(CacheAdapter):
    def __init__(self, cache):
        super().__init__(cache, 'people_service_roles')
        self.service_members_repo = get_ids_repository('abc', 'service_members', api_version=4, timeout=60)

    def prepare_services(self, lookup=None):
        services = self._get_services_by_person(lookup)
        for login, data in services.items():
            self.set(login, data)

    def _get_services_by_person(self, lookup):
        lookup = copy(lookup) if lookup else {}
        lookup.setdefault('fields', 'person.login,service.id,service.name,role.code,role.name')

        user_data = defaultdict(dict)
        for role in self.service_members_repo.getiter(lookup=lookup):
            login = role['person']['login']
            service_id = role['service']['id']

            if role['role']['code'] == 'product_head':
                user_data[login]['is_owner'] = True

            user_data[login].setdefault('services', {})
            if user_data[login]['services'].get(service_id):
                user_data[login]['services'][service_id]['roles'].append(role['role'])
            else:
                user_data[login]['services'][service_id] = {'service': role['service'],
                                                            'roles': [role['role']]}
        return user_data

    def get_default(self, short_key):
        data = self._get_services_by_person({'person__login': short_key})
        return data.get(short_key, {})


def gen_phones(value):
    value = str(value)
    yield value

    # FIXME phonenumbers не умеет нормально парсить телефоны без + в начале.
    # можно попробовать обновить версию (ISEARCH-4748) и если это не поможет - сделать им issue
    if not value.startswith('+'):
        value = '+' + value

    try:
        phone = phonenumbers.parse(value)
    except phonenumbers.NumberParseException:
        # TODO: здесь еще можно логику про если номер начинается с 8
        # и длинный и человек в определенной стране, то +7
        num = ''.join(i for i in str(value) if i.isdigit())
        yield num[-4:]
        yield num[-7:]
        yield num[-10:]
        return

    national = str(phone.national_number)

    # 9037439198
    yield national

    # 89037439198
    yield '8' + str(phone.national_number)

    # 9198
    yield national[-4:]

    # 7439198
    yield national[-7:]

    # u'8 (903) 743-91-98'
    yield phonenumbers.format_number(phone,
                                     phonenumbers.PhoneNumberFormat.NATIONAL)

    # +7 903 743-91-98
    yield phonenumbers.format_number(phone,
                                     phonenumbers.PhoneNumberFormat.INTERNATIONAL)

    # +79037439198
    yield phonenumbers.format_number(phone,
                                     phonenumbers.PhoneNumberFormat.E164)


def normalize_phone(value, region='RU'):
    """ Приводит значение телефона к единому формату
    """
    try:
        phone = phonenumbers.parse(value, region=region)
    except phonenumbers.NumberParseException:
        log.warning('Cannot parse phone: %s', value)
        return value

    return phonenumbers.format_number(phone, phonenumbers.PhoneNumberFormat.INTERNATIONAL)


# Регулярка, находящая границы перехода от букв к цифрам и наоборот
# в автомобильных номерах. В сматченную группу добавляет первый символ
# после "границы".
CARS_PLATE_BORDERS_RE = re.compile(r"((?<=\d)[^\d]|(?<=[^\d])\d)")


def get_car_nums(number):
    if not number:
        return

    ru = 'АВЕКМНОРСТУХ'
    en = 'ABEKMHOPCTYX'

    ru_en = dict(zip(ru, en))
    en_ru = dict(zip(en, ru))

    # добавляем оригинальное написание, потому что оно может быть особенным для мотоциклов
    # или иностранных номеров
    number = number.upper()
    number_forms = {number}

    for num in (number, number.replace(" ", "")):
        for alphabet in (ru_en, en_ru):
            local_version = ''.join(alphabet[l] if l in alphabet else l for l in num)
            # добавляем в форму номера версию без пробелов и с пробелами по границам букв и цифр
            number_forms.add(local_version)
            if ' ' not in local_version:
                number_forms.add(CARS_PLATE_BORDERS_RE.sub(r" \1", local_version))

    return list(number_forms)


def get_family_status(personal):
    if not personal['gender'] or not personal['family_status']:
        return ''

    family_statuses = {
        'single': {
            'male': ['не женат', 'холост', 'single'],
            'female': ['не замужем', 'незамужем', 'single'],
        },
        'married': {
            'male': ['married', 'женат'],
            'female': ['замужем', 'married'],
        }
    }
    return family_statuses[personal['family_status']][personal['gender']]


def get_location(data):
    location = {
        'description': data['location']['description'],
        'office': {},
        'floor': {},
        'table': {},
    }

    if data['location']['table']['id'] is not None:
        location['office'] = data['location']['table']['floor']['office']
        location['floor'] = {
            'number': data['location']['table']['floor'].get('number'),
            'id': data['location']['table']['floor'].get('id'),
            'name': data['location']['table']['floor'].get('name'),
        }
        location['table'] = {
            'id': data['location']['table']['id'],
            'number': data['location']['table'].get('number'),
        }
    elif data['location']['office']['id']:
        location['office'] = data['location']['office']

    return location


def get_cities():
    office_repo = get_ids_repository('staff', 'office')

    cities = {}
    for office in office_repo.getiter(lookup={'_sort': 'id', '_fields': 'city'}):
        cities[office['city']['id']] = office['city']['name']

    return cities


edu_dict = {'status': education_status, 'area': education_area}


def get_edu_forms(_type, key):
    edu = get_edu(_type, key)
    return [edu['ru']] + [edu['en']] + edu['_forms']


def get_edu(_type, key):
    edu = edu_dict[_type]
    try:
        return edu[key]
    except KeyError:
        return {'en': '', 'ru': '', '_forms': []}


def get_fingerprint(line, delim=':'):
    key = base64.b64decode(line.strip().split()[1].encode('ascii'))
    fp_plain = hashlib.md5(key).hexdigest()
    assert delim in (' ', ':')
    return delim.join(a+b for a, b in zip(fp_plain[::2], fp_plain[1::2]))


def get_contact_type_forms(type_):
    """ Получение списка всех форм написания типа контакта.
    Если тип неопределён, то пишет сообщение в лог.

    :param type_: название типа контакта
    :return: список форм написания
    """
    if type_ not in contact_labels and type_ not in contact_type:
        log.warning("Unknown contact type %s", type_)
        return [type_]
    return contact_type.get(type_, [type_]) + list(contact_labels.get(type_, {}).values())


def get_gender_forms(gender):
    return GENDER_FORMS.get(gender, [])


def flatten_groups(groups):
    """
    Получение плоского списка всех групп пользователей
    """
    flat_groups = {}
    for membership in groups:
        group = membership.get('group')
        flat_groups[group['id']] = group
        for ancestor in group.get('ancestors', []):
            flat_groups[ancestor['id']] = ancestor

    return list(flat_groups.values())
