from intranet.search.core.sources.plan.utils import (
    SERVICE_STATE_MAP,
    SERVICE_TRAFFIC_LIGHT_MAP,
    SERVICE_TRAFFIC_LIGHT_VALUE_MAP,
)
from intranet.search.core.sources.utils import Metrix, get_document_url, normalize
from intranet.search.core.swarm.api_indexer import PagedApiIndexer
from intranet.search.core.utils import get_ids_repository


class BaseSource(PagedApiIndexer):
    """
    Базовый индексатор сервисов ABC
    """
    service_repo = get_ids_repository('abc', 'service', api_version=4, timeout=5)
    service_contacts_repo = get_ids_repository('abc', 'service_contacts', api_version=4, timeout=5)

    SERVICE_FILTER = {}
    SERVICE_FIELDS = [
        'id',
        'slug',
        'state',
        'name',
        'description',
        'tags',
        'traffic_light',
        'unique_immediate_members_count',
        'unique_immediate_robots_count',
    ]
    CONTACT_FIELDS = [
        'type',
        'title',
        'content',
    ]

    def __init__(self, options):
        super().__init__(options)
        self.metrix = Metrix(
            cache=self.cache_storage,
            source='abc',
            regex=r'^/services/(?P<key>[\w\-]+)/?.*$',
        )

    def do_setup(self, **kwargs):
        self.metrix.prepare_metrix()
        super().do_setup(**kwargs)

    def do_walk(self, **kwargs):
        if self.options['keys']:
            services = [{'id': k} for k in self.options['keys']]
            for service in services:
                self.next('fetch', obj=service)
        else:
            lookup = dict(
                fields=','.join(self.SERVICE_FIELDS),
                **self.SERVICE_FILTER,
            )
            for service in self.service_repo.getiter(lookup=lookup):
                self.next('fetch', obj=service)

    def do_fetch(self, obj, **kwargs):
        if list(obj) == ['id']:
            lookup = {
                'id': obj['id'],
                'fields': ','.join(self.SERVICE_FIELDS),
            }
            obj = self.service_repo.get_one(lookup=lookup)

        if obj['state'] == 'deleted':
            self.do_delete(obj)
        else:
            obj = self.fetch_object(obj)
            self.next('create', obj=obj)

    def fetch_object(self, obj):
        lookup = {
            'service': obj['id'],
            'fields': ','.join(self.CONTACT_FIELDS),
        }
        obj['contacts'] = self.service_contacts_repo.get(lookup)
        return obj

    def get_doc_url(self, obj):
        return get_document_url('abc').format(slug=obj['slug'])

    def emit_factors(self, doc, obj):
        if SERVICE_STATE_MAP[obj['state']].get('factor'):
            doc.emit_factor(SERVICE_STATE_MAP[obj['state']]['factor'], 1)

        for measure in obj['traffic_light']:
            name = SERVICE_TRAFFIC_LIGHT_MAP.get(measure['group']['code'])
            value = SERVICE_TRAFFIC_LIGHT_VALUE_MAP.get(measure['level'])
            if name and value:
                doc.emit_factor(name, value)

        is_micro = any(t['slug'] == 'micro' for t in obj['tags'])
        doc.emit_factor('isMicro', int(is_micro))

        doc.emit_factor('serviceTeamCount', normalize(obj['unique_immediate_members_count'], 100))
        doc.emit_factor('serviceRobotsCount', normalize(obj['unique_immediate_robots_count'], 10))

        slug = obj['slug'].lower()
        str_id = str(obj['id'])
        page_views = self.metrix[slug] + self.metrix[str_id]
        doc.emit_factor('pageViews', normalize(page_views, 200))
