from intranet.search.core.sources.plan.base import BaseSource
from intranet.search.core.sources.plan.utils import normalize_url
from intranet.search.core.sources.utils import get_by_lang, get_text_content


class Source(BaseSource):
    SERVICE_FILTER = {'state__in': 'develop,supported,closed'}

    def do_create(self, obj, **kwargs):
        siteurls = [c for c in obj.get('contacts', [])
                    if c['type']['code'] == 'url_sitelink' and c['content']]

        if not siteurls:
            super().do_create(obj)
        else:
            for contact in siteurls:
                contact['service'] = obj
                super().do_create(contact)

    def is_contact(self, obj):
        """ Возвращает true, если объект - это контакт сервиса, а не сам сервис
        """
        return 'content' in obj

    def get_service(self, obj):
        return obj['service'] if self.is_contact(obj) else obj

    def get_doc_url(self, obj):
        """ Возвращает url документа для индексации
        """
        if self.is_contact(obj):
            return normalize_url(obj['content'])
        return super().get_doc_url(obj)

    def create_body(self, obj, **kwargs):
        service = self.get_service(obj)
        body = {
            'service_full_name': {
                'title': service['name']['ru'],
                'title_en': service['name']['en'],
                'name': obj['title'] if self.is_contact(obj) else None,
            },
            'description': get_text_content(service['description']['ru']),
        }
        return body

    def create_snippet(self, obj, lang='ru', **kwargs):
        service = self.get_service(obj)
        snippet = {
            'title': (get_by_lang(obj['title'], lang) if self.is_contact(obj) else None) or None,
            'service_name': get_by_lang(service['name'], lang),
            'href': self.get_doc_url(obj),
            'slug': service['slug'],
            'abc': not self.is_contact(obj),
        }
        return snippet

    def emit_factors(self, doc, obj):
        super().emit_factors(doc, self.get_service(obj))

    def emit_attrs(self, doc, obj):
        service = self.get_service(obj)
        slug = service['slug']
        doc.emit_suggest_attr(slug)
        doc.emit_suggest_attr(service['name']['ru'])
        doc.emit_suggest_attr(service['name']['en'])

        if self.is_contact(obj):
            doc.emit_suggest_attr(obj['title']['ru'])
            doc.emit_suggest_attr(obj['title']['en'])
