import logging

from collections import defaultdict

from django.conf import settings
from django import db
from django.utils.dateparse import parse_datetime
from django.utils.functional import cached_property
from requests import HTTPError
from startrek_client import Startrek
from startrek_client.exceptions import StartrekError, NotFound
from startrek_client.objects import PaginatedList

from intranet.search.core.snippets.st import IssueSnippet, QueueSnippet
from intranet.search.core.sources.st.utils import (
    sanitized_text,
    display,
    get_group_attrs_for_factor_by_uid,
    AttrDict,
    is_issue_in_blacklist,
)
from intranet.search.core.sources.utils import (
    get_document_url,
    truncate_chars,
    date_to_timestamp,
    timestamp_to_utc_date,
    get_suggest_parts,
    get_by_lang,
)
from intranet.search.core.sources.st.utils import get_issue_number
from intranet.search.core.storages.cache import GlobalCacheStorage
from intranet.search.core.swarm import Indexer
from intranet.search.core.tvm import tvm2_client
from intranet.search.core.utils import reraise_as_recoverable


log = logging.getLogger(__name__)


class BaseTrackerIndexer(Indexer):

    api_settings = settings.ISEARCH['api']['st']
    repo_params = {
        'timeout': 10,
    }

    @property
    def tvm_ticket(self):
        ticket = tvm2_client.get_service_ticket('tracker')
        return ticket

    def _get_api_client(self):
        endpoint = self.api_settings['endpoint']
        return Startrek(
            base_url=endpoint.url(),
            useragent=endpoint.get('headers').get('User-Agent'),
            service_ticket=self.tvm_ticket,
            user_ticket='',
            api_version='service',
            **self.repo_params
        )

    @cached_property
    def api_client(self):
        return self._get_api_client()

    @property
    def issues(self):
        return self.api_client.issues

    @property
    def queues(self):
        return self.api_client.queues

    @property
    def components(self):
        return self.api_client.components

    def get_users_permissions(self, permissions):
        return [u['id'] for u in permissions['users']]

    def get_groups_permissions(self, permissions):
        return [int(g['id']) for g in permissions['groups']]

    def get_permissions(self, raw_permissions, scopes=None):
        scopes = scopes or []
        plain_permissions = defaultdict(list)

        for scope in scopes:
            perm = raw_permissions.get(scope) or {}
            if not perm:
                continue
            for perm_type in ('groups', 'users'):
                plain_permissions[perm_type].extend(perm.get(perm_type, []))

        return {
            'groups': self.get_groups_permissions(plain_permissions),
            'users': self.get_users_permissions(plain_permissions),
        }

    def add_permissions(self, doc, permissions):
        """
        Добавляет пермишшены в документ
        """
        for group in permissions['groups']:
            doc.emit_search_attr('acl_groups_whitelist', str(group))
        for user in permissions['users']:
            doc.emit_search_attr('acl_users_whitelist', user)


class BaseTrackerIssuesIndexer(BaseTrackerIndexer):

    empty_data = {'ru': 'Нет', 'en': 'None'}
    issues_expand_params = 'permissions,html'

    # дополнительные атрибуты для поиска в заивисмости от конкретной очереди
    QUEUE_SEARCH_ATTRS = {}

    def __init__(self, options):
        super().__init__(options)
        self.global_cache_storage = GlobalCacheStorage()

        self.priority_map = {
            'blocker': 1.,
            'critical': 0.8,
            'normal': 0.6,
            'trivial': 0.4,
            'minor': 0.2,
        }

        log.debug('ST source initialized')

    def parse_keys(self, keys):
        queues = set()
        components = set()
        tickets = set()

        for key in keys:
            if key.startswith('#'):
                components.add(key[1:])
            elif '-' in key:
                tickets.add(key)
            else:
                queues.add(key)

        return {'queues': queues, 'components': components, 'tickets': tickets}

    def do_setup(self, **kwargs):
        if self.options['from_cache']:
            self.next('load')
            return

        keys = self.parse_keys(self.options['keys'])
        queues = keys['queues']
        components = keys['components']
        tickets = keys['tickets']

        if not queues and not components and not tickets:
            log.debug('Get all issues')
            self.next('walk')

        components = self.clean_components(queues, components)

        for queue in queues:
            log.debug('Get queue: %s', queue)
            self.next('walk', query={'queue': queue})

        for component in components:
            log.debug('Get component: %s', component)
            self.next('walk', query={'components': component})

        for ticket in tickets:
            log.debug('Get ticket: %s', ticket)
            self.next('fetch', issue={'key': ticket}, refetch_issue=True)

    def clean_components(self, queues, components):
        """ Исключаем из индексации компоненты, очередь которых и так есть в queues
        """
        if not queues:
            return components

        new_components = set()
        for comp_id in components:
            try:
                comp_queue = self.components[comp_id].queue.key
            except NotFound:
                log.debug('Component does not exists: %s', comp_id)
                continue
            except Exception:
                log.exception('Got exception while check component: %s', comp_id)
                comp_queue = None
            if comp_queue not in queues:
                new_components.add(comp_id)

        log.debug('Remain components: %s', new_components)
        if components - new_components:
            log.debug('Skip components: %s', components - new_components)

        return new_components

    def get_filter_params(self, query=None, page=1, per_page=100):
        params = dict(staleOk='true', page=page, per_page=per_page, order=['key'])

        params['filter'] = {}
        params['filter'].update(query or {})
        if self.options['ts']:
            updated = timestamp_to_utc_date(self.options['ts'])
            params['filter']['updated'] = {'from': updated.isoformat()}
        return params

    def objects_count(self, updated_since=None, updated_till=None):
        query = {'updated': {'from': updated_since.isoformat(), 'to': updated_till.isoformat()}}
        params = self.get_filter_params(query, page=1, per_page=1)
        log.debug('Call st api with params: %s', params)
        objects = self.issues.find(**params)
        return objects._items_count if hasattr(objects, '_items_count') else len(objects)

    @reraise_as_recoverable(StartrekError, db.Error)
    def do_walk(self, query=None, from_id=None, **kwargs):
        per_page = 100
        query = query or {}
        params = self.get_filter_params(query, per_page=per_page)
        params.update(expand=self.issues_expand_params, localized='false')

        only_one_queue_issues = bool({'queue', 'components'} & query.keys())
        if only_one_queue_issues:
            params['fromNum'] = from_id or 0
        elif from_id and isinstance(from_id, str):
            params['from_id'] = from_id

        log.debug('Call st api with params: %s', params)
        objects = self.issues.relative_scroll(**params)

        if isinstance(objects, PaginatedList):
            objects = objects._data

        if objects and len(objects) == per_page:
            last_issue = objects[-1].as_dict()
            # next_from_id может быть:
            # 1. id последнего тикета, если очередь не определена,
            # например, при индексации организации и Трекера целиком;
            # 2. номером из ключа последнего тикета, если очередь определена,
            # например, при индексации отдельной очереди или компонента.
            if only_one_queue_issues:
                next_from_id = get_issue_number(last_issue)
            else:
                next_from_id = last_issue['id']
            self.next('walk', query=query, from_id=next_from_id)

        for issue in objects:
            self.next('fetch', issue=issue.as_dict())

    @reraise_as_recoverable(StartrekError, db.Error, HTTPError)
    def do_fetch(self, issue, refetch_issue=False, **kwargs):
        if is_issue_in_blacklist(issue['key']):
            log.warning('Issue %s is in black list. Skip indexing.', issue['key'])
            return

        if refetch_issue:
            issue = self.issues.get(issue.get('key'), expand=self.issues_expand_params,
                                    localized='false').as_dict()

        issue = AttrDict(issue)
        log.debug('ST fetch: %s' % issue.key)

        permissions = self.get_permissions(issue.permissions, ['read'])
        queue = issue.queue
        try:
            attachments = self.issues.attachments(issue).get_all()
        except Exception as e:
            log.error(e)
            attachments = []

        # берем только 200 комментариев, не пытаемся вытащить их все
        comments = self.issues.comments(issue).get_all(expand='html', staleOk=True, perPage=200)
        if isinstance(comments, PaginatedList):
            comments._one_page = True
        self.extra_fetch(issue)

        self.next(
            stage='create',
            issue=dict(issue),
            queue=queue,
            permissions=permissions,
            comments=[c.as_dict() for c in comments],
            attachments=[a.as_dict() for a in attachments],
        )

    def extra_fetch(self, issue):
        pass

    def do_create(
        self, issue, queue={}, permissions={}, comments=[],
        attachments=[], delete=False, **kwargs
    ):
        if delete:
            self.delete_issue(issue)
        else:
            self.create_issue(issue, queue, permissions, comments, attachments)
        for old_key in issue.get('aliases', []):
            # для перенесенных задач удаляем все записи по старым ключам
            self.delete_issue({'key': old_key})

    def delete_issue(self, issue):
        doc_url = get_document_url('st').format(key=issue['key'])
        doc = self.create_document(doc_url)
        if self.need_content:
            self.next('content', url=doc_url, delete=True)
        self.next('store', document=doc, delete=True)

    def create_issue(self, issue, queue, permissions, comments, attachments):
        log.debug('ST create: %s' % issue['key'])

        # кеш для быстрой переиндексации
        cache = {'issue': issue, 'queue': queue,
                 'permissions': permissions, 'comments': comments}

        issue = AttrDict(issue)
        queue = AttrDict(queue)
        comments = [AttrDict(c) for c in comments]

        assignee = self.get_person_data(issue.assignee)
        author = self.get_person_data(issue.createdBy)
        components = [self.get_data(comp) for comp in (issue.components or [])]
        status = self.get_data(issue.status)

        if issue.project:
            project = {
                'id': issue['project']['id'],
                'name': display(issue.project, 'ru'),
                'name_en': display(issue.project, 'en'),
            }
        else:
            project = {'name': '', 'name_en': '', 'id': None}

        doc_url = get_document_url('st').format(key=issue.key)
        doc = self.create_document(doc_url, updated=parse_datetime(issue.updatedAt))

        doc.emit_property_attr('key', issue.key)

        # групповые атрибуты для пользовательских факторов
        doc.emit_group_attr('queue', int(queue.id))
        self.emit_user_factor(doc, 'author', issue.createdBy)
        if issue.assignee:
            self.emit_user_factor(doc, 'assignee', issue.assignee)

        self.add_permissions(doc, permissions)

        for suggest_attr in (issue.key, issue.summary):
            doc.emit_suggest_attr(suggest_attr)
            for part in get_suggest_parts(suggest_attr):
                doc.emit_suggest_attr(part)

        self.emit_facet_attrs(doc, issue, queue)

        is_closed = 1 if status['key'] == 'closed' else 0

        updated = date_to_timestamp(issue.updatedAt)
        created = date_to_timestamp(issue.createdAt)
        doc.emit_search_attr('i_updated', updated)

        doc.emit_factor('UpdatedAt', updated)
        doc.emit_factor('CreatedAt', created)
        doc.emit_factor('isClosed', is_closed)
        doc.emit_factor('Priority',
                        self.priority_map.get(issue['priority']['key'], 0))

        if issue.resolution and issue.resolution['key'] == 'duplicate':
            is_duplicate = 1
        else:
            is_duplicate = 0

        doc.emit_factor('isDuplicated', is_duplicate)

        doc.emit_factor('isResolved', int(bool(issue.resolution or is_closed)))

        if queue.key in ('TEST', 'GRAPHCONTENT'):
            doc.emit_factor('isTest', 1)

        description = sanitized_text(issue, 'descriptionHtml', 'description') or ''
        additional_attributes = self.get_additional_attributes(issue)

        body = {
            'description': description or '',
            'comments': [],
            '!hidden': {
                'summary': issue.summary,
                'author': author,
                'queue': {
                    'key': queue.key,
                    'name': display(queue, 'ru'),
                    'ticket': issue.key,
                },
                'components': [c['name'] for c in components],
                'tags': issue.tags,
                'project': {project['name'], project['name_en']},
                'aliases': issue.aliases,
                'additional': additional_attributes,
                'attachments': [a['name'] for a in attachments],
            },
        }
        for comment in comments:
            body['comments'].append(sanitized_text(comment, 'textHtml', 'text'))
            if 'email' in comment:
                body['comments'].append(sanitized_text(comment['email'], 'text'))

        if issue.assignee:
            body['!hidden']['assignee'] = assignee

        doc.emit_body(body)

        short_description = truncate_chars(description, 250)

        def build_snippet(lang):
            additional = {}
            for key, value in additional_attributes.items():
                additional[key] = get_by_lang(value, lang)

            attachments_data = []
            for attachment in attachments:
                attachments_data.append({
                    'id': attachment['id'],
                    'name': attachment['name'],
                    'url': get_document_url('tracker_attachment').format(
                        key=issue.key,
                        id=attachment['id'],
                    ),
                    'created': attachment['createdAt'],
                    'mimetype': attachment['mimetype'],
                })

            return IssueSnippet({
                'queue': self.get_data(queue),
                'key': issue.key,
                'summary': issue.summary,
                'components': components,
                'tags': issue.tags,
                'assignee': self.get_person_data(issue.assignee, lang),
                'priority': self.get_data(issue.priority, lang),
                'description': short_description or '',
                'status': self.get_data(issue.status, lang),
                'resolution': self.get_data(issue.resolution, lang),
                'author': self.get_person_data(issue.createdBy, lang),
                'created': created,
                'updated': updated,
                'due_date': date_to_timestamp(issue.deadline) if issue.deadline else None,
                'project': project,
                'services': [
                    {'id': s['id'], 'name': get_by_lang(s['name'], lang)}
                    for s in issue.get('services', [])
                ],
                'additional': additional_attributes,
                'attachments': attachments_data,
            })

        doc.emit_snippet(build_snippet('ru'), 'ru')
        doc.emit_snippet(build_snippet('en'), 'en')

        if self.need_content:
            self.next('content', url=doc.url, raw_data=cache, updated_ts=doc.updated_ts)
        self.next('store', document=doc)

    def emit_facet_attrs(self, doc, issue, queue):
        doc.emit_facet_attr('queue', queue.key, queue.key, queue.key)
        doc.emit_facet_attr(
            name='type',
            value=issue.type['key'],
            label=display(issue.type, 'ru'),
            label_en=display(issue.type, 'en'),
        )
        for field in ('resolution', 'status', 'assignee', 'author'):
            data = issue.get('createdBy' if field == 'author' else field)
            ru_data = self.get_data(data, 'ru')
            en_data = self.get_data(data, 'en')
            doc.emit_facet_attr(field, ru_data['id'], ru_data['name'], en_data['name'])

    def get_uid(self, data):
        """ Получаем uid пользователя. Его можно достать из урла на апи пользователя (поле self).
        Это не очень прямой способ, но более быстрый, чем дерганье ещё и апи стаффа/директории.
        """
        return int(data['self'].split('/')[-1])

    def emit_user_factor(self, doc, name, data):
        uid = self.get_uid(data)
        for attr_name, attr_value in get_group_attrs_for_factor_by_uid(uid, name):
            doc.emit_group_attr(attr_name, attr_value)

    def get_person_data(self, data, lang='ru'):
        if not data:
            return {'uid': None, 'login': 'none', 'name': self.empty_data[lang]}

        return {
            'uid': self.get_uid(data),
            'login': data['id'],
            'name': display(data, lang)
        }

    def get_data(self, data, lang='ru'):
        if not data:
            return {'key': 'none', 'name': self.empty_data[lang], 'id': 'none'}

        name = display(data, lang)
        return {'key': data.get('key'), 'name': name, 'id': data['id']}

    def get_additional_attributes(self, issue):
        """ Возвращает дополнительные атрибуты для поиска в заивисмости
        от конкретной задачи
        """
        extra_attrs = {}
        for attr_name in self.QUEUE_SEARCH_ATTRS.get(issue.queue['key'], []):
            attr_value = issue.get(attr_name)
            if attr_value is None:
                continue
            if isinstance(attr_value, list):
                attr_value = ', '.join(display(v) for v in attr_value)
            else:
                attr_value = display(attr_value)
            extra_attrs[attr_name] = attr_value
        return extra_attrs

    def do_push(self, data, delete=False, **kwargs):
        id_ = data['meta']['issueKey']

        if delete:
            self.delete_issue({'key': id_})
        else:
            self.next('fetch', issue={'key': id_}, refetch_issue=True)


class BaseTrackerQueuesIndexer(BaseTrackerIndexer):
    """
    Базовый индексатор очередей стартрека
    """
    def do_walk(self, **kwargs):
        if self.options['keys']:
            for k in self.options['keys']:
                self.next('fetch', key=k)
        else:
            for q in self.queues.get_all():
                self.next('fetch', obj=q)

    def do_fetch(self, obj=None, key=None, **kwargs):
        if key:
            obj = self.queues.get(key)
        if not obj:
            log.warning('No queue for fetching. Key: %s', key)
            return

        permissions = self.get_permissions(
            raw_permissions=obj.permissions.as_dict(),
            scopes=['create', 'read', 'write', 'writeNoAssign'],
        )
        self.next('create', obj=obj.as_dict(), permissions=permissions)

    def do_create(self, obj, permissions, **kwargs):
        doc_url = get_document_url('st').format(key=obj['key'])
        doc = self.create_document(doc_url)

        body = {
            'queue_key': obj['key'],
            'queue_name': obj['name'],
            'queue_description': obj.get('description', ''),
        }
        doc.emit_body(body)
        for lang in ('ru', 'en'):
            doc.emit_snippet(self.create_snippet(obj, lang), lang)

        for attr in ('key', 'name'):
            doc.emit_suggest_attr(obj[attr])

        self.add_permissions(doc, permissions)

        # для пользовательского фактора
        doc.emit_group_attr('queue', int(obj['id']))
        self.next('store', document=doc)

    def create_snippet(self, obj, lang='ru'):
        data = {
            'key': obj['key'],
            'url': self.get_doc_url(obj),
            'title': obj['name'],
            'description': obj.get('description', ''),
        }
        return QueueSnippet(data)

    def get_doc_url(self, obj):
        return get_document_url('st').format(key=obj['key'])

    def do_push(self, data, **kwargs):
        key = data['subject']['id']
        self.options['keys'] = [key]
        self.next('fetch', key=key)
