from datetime import datetime
from logging import getLogger
from urllib.parse import urljoin
from ylog.context import log_context

from django.conf import settings
from django.utils.dateparse import parse_datetime
from django.utils.functional import cached_property
from django.utils.timezone import get_current_timezone

from intranet.search.core.swarm.api_indexer import PagedApiIndexer
from intranet.search.core.utils import reraise_as_recoverable, http
from intranet.search.core.sources.utils import (
    timestamp_to_utc_date,
    date_isoformat,
    date_as_factor,
    get_person,
    get_person_details,
    normalize,
    StaffApiClientError
)

log = getLogger(__name__)


class StackoverflowApiClient:
    page_size_default = 100
    base_api = settings.ISEARCH['api']['stackoverflow']

    @cached_property
    def session(self):
        session = http.create_session()
        session.headers.update(self.base_api['endpoint'].headers())
        return session

    def fetch(self, url, **kwargs):
        response = http.call_with_retry(self.session.get, url,
                                        **kwargs)  # Будет ли автоматически декодировать gzip формат? Вроде да
        return response.json()

    def fetch_page(self, url, page=1, pagesize=page_size_default, filter_data='default', order='desc',
                   sort_by='activity'):
        """ Извлекает одну из страниц коллекции
        """
        params = {
            'page': page,
            'pagesize': pagesize,
            'filter': filter_data,
            'order': order,
            'sort': sort_by,
        }

        return self.fetch(url, params=params)

    def get_object_withbody(self, source_url, obj_id):
        params = {'filter': 'withbody'}
        return self.fetch(urljoin(source_url, str(obj_id)), params=params)

    def get_user(self, user_id):
        params = {'filter': self.base_api['users']['filter']}
        return self.fetch(urljoin(self.base_api['users'].url(), str(user_id)), params=params)['items'][0]


class SourceBase(PagedApiIndexer):
    """ Базовый индексатор для Stackoverflow
    """
    # название источника для индексации, переопределяется в наследниках
    source = ''
    # название в единственном числе (используется как атрибут в апи)
    source_singular = ''
    # человекочитаемые нахвания типа источника
    source_label = ''
    source_label_plural = ''
    # префикс для названий статический факторов
    stat_factor_prefix = ''

    # дата обновления объекта по умолчанию, отправляется в документ,
    # если получили объект без дат
    default_updated = get_current_timezone().localize(datetime.fromtimestamp(0))

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.api_client = StackoverflowApiClient()
        self.source_url = settings.ISEARCH['api']['stackoverflow'][self.source].url()

    @reraise_as_recoverable(*http.ERRORS)
    def do_walk(self, page=1, **kwargs):
        objects, has_more = self.fetch_objects(page)
        last_page = False

        for obj in objects:
            if self.options['ts'] and obj['last_activity_date'] < self.options['ts']:
                last_page = True
            func = 'fetch' if self.need_fetch else 'create'
            self.next(func, obj=obj)

        if has_more and not last_page:
            self.next('walk', page=page + 1)

    def fetch_object(self, obj_id):
        return self.api_client.get_object_withbody(self.source_url, obj_id)['items'][0]

    def fetch_objects(self, page=1):
        fetched_page = self.fetch_page(page)
        return (obj for obj in fetched_page['items']), self.has_more_pages(fetched_page)

    def fetch_page(self, page=1):
        return self.api_client.fetch_page(self.source_url, page)

    def get_doc_updated(self, obj):
        with log_context(obj=obj):
            log.info("What we've got from stackoverflow")
        date = obj.get('last_activity_date') or obj.get('creation_date', '')
        if not date:
            obj_type = self.source_singular + '_type'
            obj_id = self.source_singular + '_id'
            log.warning(f'Got object without date: {obj[obj_type]}, {obj[obj_id]}')
        return date_isoformat(timestamp_to_utc_date(date)) or self.default_updated

    def get_doc_creation_date(self, obj):
        return date_isoformat(timestamp_to_utc_date(obj['creation_date']))

    def get_doc_url(self, obj):
        return obj.get('link')

    def has_more_pages(self, data):
        return data['has_more']

    def emit_factors(self, doc, obj):
        """ Добавление факторов к документу
        """
        doc.emit_factor('updated', date_as_factor(parse_datetime(self.get_doc_updated(obj))))
        doc.emit_factor('created', date_as_factor(parse_datetime(self.get_doc_creation_date(obj))))

        owner = obj['owner']

        doc.emit_factor('reputation_author', normalize(owner['reputation'], 150))
        doc.emit_factor('user_type', int(owner['user_type'] != 'registered'))

        try:
            person_data = get_person_details(owner['display_name'])
        except StaffApiClientError:
            try:
                user_email = self.api_client.get_user(owner['user_id'])['email']
                person_data = get_person({'work_email': user_email})
            except KeyError:
                with log_context(obj=obj):
                    log.error('User on stackoverflow has no email')
                    return
            except StaffApiClientError:
                with log_context(obj=obj):
                    log.error('Can`t find user by login and email')
                    return

        is_dismissed = person_data['official']['is_dismissed']
        doc.emit_factor('isFormer', int(is_dismissed))

    def emit_attrs(self, doc, obj):
        """ Добавление атрибутов к документу
        """
        pass

    def create_snippet(self, obj, lang='ru', **kwargs):
        pass

    def create_body(self, obj, **kwargs):
        pass
