import html
from logging import getLogger

from intranet.search.core.snippets.stackoverflow import StackOverFlowSnippet
from intranet.search.core.sources.stackoverflow.base import SourceBase
from intranet.search.core.sources.utils import normalize, get_text_content, truncate_chars

log = getLogger(__name__)


class Source(SourceBase):
    """ Индексатор вопросов
    """
    source = 'questions'
    source_singular = 'question'
    source_label = 'вопрос'
    source_label_plural = 'вопросы'
    need_fetch = True

    def do_fetch(self, obj, **kwargs):
        obj_id = obj['question_id']
        obj = self.fetch_object(obj_id)
        self.next('create', obj=obj)

    def create_snippet(self, obj, lang='ru', **kwargs):
        description = html.unescape(get_text_content(obj['body']))
        short_description = truncate_chars(description, 500)
        snippet = {
            'url': self.get_doc_url(obj),
            'title': html.unescape(obj['title']),
            'description': short_description,
            'updated': self.get_doc_updated(obj),
            'is_answered': obj['is_answered'],
            'tags': obj['tags'],
            'answer_count': obj['answer_count'],
            'score': obj['score'],
        }
        return StackOverFlowSnippet(snippet)

    def create_body(self, obj, **kwargs):
        # TODO: add answers and comments
        description = html.unescape(get_text_content(obj['body']))
        body = {
            'body': description,
            'z_ns_hidden': {
                'tags': obj['tags'],
                'title': html.unescape(obj['title']),
            }
        }

        return body

    def emit_factors(self, doc, obj):
        super().emit_factors(doc, obj)
        doc.emit_factor('has_answers', normalize(obj['answer_count'], 3))
        doc.emit_factor('is_answered', int(obj['is_answered']))
        doc.emit_factor('score', normalize(obj['score'], 50))

    def emit_attrs(self, doc, obj):
        for tag in obj.get('tags', []):
            doc.emit_facet_attr('tags', tag, label=tag, label_en=tag)

        has_answers_ru, has_answers_en = ('Да', 'Yes') if obj['answer_count'] else ('Нет', 'No')
        is_answered_ru, is_answered_en = ('Да', 'Yes') if obj['is_answered'] else ('Нет', 'No')
        doc.emit_facet_attr('has_answers', has_answers_ru, label=has_answers_ru, label_en=has_answers_en)
        doc.emit_facet_attr('is_answered', is_answered_ru, label=is_answered_ru, label_en=is_answered_en)
