import calendar
import os
import re
import logging
from collections import Counter
from copy import deepcopy
from datetime import date, datetime, timedelta
from hashlib import md5
from time import mktime
from urllib.parse import urljoin

import json
import requests

from dateutil.parser import parse
from django.utils import timezone
from lxml import etree
from lxml.html import clean

from django.conf import settings

from ids.exceptions import BackendError

from intranet.search.core.errors import UnrecoverableError
from intranet.search.core.utils import get_ids_repository
from intranet.search.core.utils.cache import CacheAdapter
from intranet.search.core.utils.xml import parse_html

log = logging.getLogger(__name__)


def find_one(parsed, path, default=''):
    try:
        return parsed.xpath(path)[0]
    except IndexError:
        return default


def get_breadcrumbs(breadcrumbs, base_url=None):
    breadcrumbs_list = []
    breadcrumbs_parsed = parse_html(breadcrumbs)
    for bc in breadcrumbs_parsed.xpath('//bc'):
        url = find_one(bc, './url').text
        if base_url and not url.startswith('http'):
            url = urljoin(base_url, url)
        breadcrumbs_list.append({
            'name': find_one(bc, './name').text,
            'url': url,
        })
    return breadcrumbs_list


cleaner = clean.Cleaner(scripts=True, comments=True, style=True,
                        javascript=True, links=False,
                        remove_tags=['autocut'], kill_tags=['autocut-text'])


def sanitize(content):
    try:
        cleaner(content)
    except Exception:
        log.warning('Cannot sanitize content')
    return content


BREAK_WORD = {'li', 'a'}
BREAK_PARAGRAPH = {'ul', 'div', 'p', 'article', 'h1', 'h2', 'h3', 'h4', 'h5', 'br'}


def get_breaking_element(tag_name):
    if tag_name in BREAK_PARAGRAPH:
        return '\n'
    if tag_name in BREAK_WORD:
        return ' '


def go_through_children(xml):
    all_text = []
    br_el = get_breaking_element(xml.tag)
    if br_el:
        all_text.append(br_el)

    if xml.text:
        all_text.append(xml.text)
    for child in xml.iterchildren():
        all_text.extend(go_through_children(child))

    if br_el:
        all_text.append(br_el)

    if xml.tail:
        all_text.append(xml.tail)

    return all_text


def get_text_content(data, sanitize_=True):
    """ Парсит html строчку и возвращает строчку без тегов, только текст

    data - html строка или etree.Element
    """
    if isinstance(data, str):
        if not data.strip():
            return ''

        data = data.replace('\n', '<br/>')
        try:
            xml = parse_html(data)
        except (etree.ParserError, etree.XMLSyntaxError):
            xml = None
    else:
        # уже все распаршено
        xml = data

    if xml is None:
        content = str(data)
    else:
        if sanitize_:
            xml = sanitize(xml)

        content = ''.join(go_through_children(xml))
        content = re.sub(r'\s*\n+\s*', r'\n', content).strip()
        content = re.sub(r'[ \t]+', r' ', content).strip()
    return content


def get_elements_content(content, tags):
    """ Возвращает список с текстовым контентом всех указанных элементов
    """
    result = []
    for tag in tags:
        for el in content.xpath(f'//{tag}'):
            text = get_text_content(el)
            if text:
                result.append(text)
    return result


def split_by_sub_titles(data):
    """
    Разбивает контент документа на части по заголовкам h*, у которых есть id (то есть тех,
    на которые можно получить якорь). Первым элементом складывает вступительную часть.
    :param data: элемент etree.Element
    """
    anchor_tags = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6']

    sub_titles = data.xpath('|'.join(f'//{tag}[@id]' for tag in anchor_tags))
    if not sub_titles:
        return []

    results = []
    content = sub_titles[0].getparent()
    unanchored_content = etree.Element('div')

    for element in content:
        if element in sub_titles:
            break
        unanchored_content.append(deepcopy(element))

    if list(unanchored_content):
        results.append({'title': None, 'content': unanchored_content})

    for title in sub_titles:
        content = etree.Element('div')
        content.append(deepcopy(title))

        next_el = title.getnext()
        while next_el is not None and next_el not in sub_titles:
            content.append(deepcopy(next_el))
            next_el = next_el.getnext()

        results.append({'title': title, 'content': content})

    return results


def get_document_url(source):
    return settings.ISEARCH['urls'][source]


def get_resource_data(resource):
    if isinstance(resource, dict):
        return resource
    elif isinstance(resource, (str, bytes)):
        return json.loads(resource)
    else:
        raise ValueError('Wrong resource type, got %s', type(resource))


def get_metrix_data(source):
    """ Получение данных от метрики о количестве просмотров
    страниц заданного источника

    :param source: имя источника (wiki, lego, etc.)
    :return: генератор значений вида {'path': '/brickbox/', 'page_views': 97}
    """
    # date1 - с какого момента считать статистику, формат YYYYMMDD
    date1 = (datetime.now() - timedelta(days=30)).strftime('%Y%m%d')

    url = settings.ISEARCH['api']['metrix'][source].url(query={'date1': date1})
    headers = settings.ISEARCH['api']['metrix'][source].headers()
    resp = requests.get(url, headers=headers)

    if not resp.ok:
        raise Exception('Metrix response was not OK')

    for d in resp.json()['data']:
        yield {'path': d['dimensions'][0]['name'], 'page_views': d['metrics'][0]}


def date_as_factor(timestamp_or_date, norm_date=date(2020, 1, 1)):
    norm = mktime(norm_date.timetuple())
    if timestamp_or_date is None:
        return 0
    elif isinstance(timestamp_or_date, (int, float)):
        return timestamp_or_date / norm
    elif isinstance(timestamp_or_date, (date, datetime)):
        return date_to_timestamp(timestamp_or_date) / norm
    elif isinstance(timestamp_or_date, str):
        return date_as_factor(float(timestamp_or_date))
    else:
        raise TypeError("Unknown type %s of argument" % type(timestamp_or_date))


def date_to_timestamp(date_factor):
    """ Переводит дату в timestamp, которые могут использоваться в качестве числового атрибута
    """
    if isinstance(date_factor, str):
        date_factor = parse(date_factor)

    assert isinstance(date_factor, datetime)
    # не используем mktime, т.к. он переводит в локальное время, а не в utc
    return int(calendar.timegm(date_factor.utctimetuple()))


def timestamp_to_utc_date(timestamp):
    """ Переводим unixtimestamp в дату с указанным tzinfo
    """
    dt = datetime.utcfromtimestamp(float(timestamp)).replace(tzinfo=timezone.utc)
    return dt


def date_isoformat(d):
    fmt = '%Y-%m-%dT%H:%M:%S%z'
    return d.strftime(fmt)


def hash_factor(string, precision=10):
    """ Фактор, добавляющий уникальности в значение релевантности при прочих равных
    :return: псевдо-уникальное число от 0 до 1
    """
    return float('0.%s' % int(md5(string.encode('utf-8')).hexdigest()[:precision], 16))


staff_repo = get_ids_repository('staff', 'person', timeout=settings.ISEARCH['abovemeta']['timeouts']['staffapi'])


class StaffApiClientError(UnrecoverableError):
    pass


def get_persons(lookup=None):
    if lookup is None:
        lookup = {'_sort': 'id', '_fields': 'id,login'}
    return staff_repo.get_nopage(lookup=lookup)


def get_person(lookup):
    try:
        return staff_repo.get_one(lookup=lookup)
    except BackendError as exc:
        if 400 <= exc.status_code < 500:
            raise StaffApiClientError from exc
        raise exc


def get_person_details(login):
    return get_person({'login': login})


def normalize_email(email_string):
    """ Нормализуем email-адрес в полный и корректный
    """
    result = None
    if email_string:
        # ошибки в нормализации не должны ломать индексацию, поэтому завернуто в try
        try:
            # если перечислено несколько адресов через запятую, берем самый первый.
            first_email = email_string.split(',')[0].strip()
            email_tuples = first_email.split('@')
            if len(email_tuples) > 1 and email_tuples[1]:
                result = email_tuples[0] + '@' + email_tuples[1]
            else:
                # Если домен не указан, ставим @yandex-team.ru
                result = email_tuples[0] + '@yandex-team.ru'
        except:
            pass
    return result


def get_person_fio(data, lang, middle=True):
    fio = [get_by_lang(data['name']['last'], lang),
           get_by_lang(data['name']['first'], lang)]
    if lang == 'ru' and middle:
        fio.append(data['name'].get('middle', ''))
    return ' '.join(fio)


def get_by_lang(data, lang):
    if not data:
        return ''
    if not isinstance(data, dict):
        return data
    another_lang = 'en' if lang == 'ru' else 'ru'
    return data.get(lang) or data.get(another_lang) or ''


def get_person_name_by_lang(data, lang):
    name = {}
    for name_part in ('first', 'last', 'middle'):
        name[name_part] = get_by_lang(data['name'].get(name_part), lang)
    return name


def get_short_desc(description, desc_len):
    if len(description) > desc_len:
        # находим первый пробел после desc_len символов и обрубаем по него
        description = description[:description[desc_len:].find(' ') + desc_len]
        if description[-1] != '.':
            description = '%s...' % description
    return description


def normalize(value, norm):
    if not value:
        return 0.
    elif float(value) > norm:
        return 1.
    else:
        return float(value) / norm


def fix_encoding(raw_content, parsed):
    ctype = parsed.xpath('/html/head/meta[@http-equiv="Content-Type"]')
    # если кодировка была правильно передана - то parsed подходит
    if ctype:
        return parsed

    charset = find_one(parsed, '/html/head/meta/@charset', None)
    # если есть какие-то знания про кодировку - используем их
    if charset:
        return parse_html(raw_content.decode(charset))

    # иначе пробуем utf8
    return parse_html(raw_content.decode('utf-8'))


LAYOUT_RU = 'йцукенгшщзхъфывапролджэёячсмитьбюЙЦУКЕНГШЩЗХЪФЫВАПРОЛДЖЭЁЯЧСМИТЬБЮ'
LAYOUT_EN = 'qwertyuiop[]asdfghjkl;\'\\zxcvbnm,.QWERTYUIOP{}ASDFGHJKL:\"|ZXCVBNM<>'
LAYOUTS = {
    'ru': {ord(k): ord(v) for k, v in zip(LAYOUT_RU, LAYOUT_EN)},
    'en': {ord(k): ord(v) for k, v in zip(LAYOUT_EN, LAYOUT_RU)},
}


def swap_layout(text, orig):
    """ Меняет раскладку текста с orig на to_lang
    """
    if not isinstance(text, str):
        return ''
    return text.translate(LAYOUTS[orig])


def truncate_chars(value, limit, truncate='...'):
    """ Обрезает N символов текста, оставляя при этом последнее слово необрезанным
    """
    if not value:
        return ''

    value = str(value)

    if len(value) <= limit:
        return value

    value = value[:limit]
    return value.rsplit(' ', 1)[0] + truncate


def get_content_type(response):
    """ Возвращает чистое значение content_type ответа

    :param response: объект requests.http.Response
    :return: строка content-type
    """
    return response.headers.get('content-type').split(';')[0]


def get_suggest_parts(value, min_suggest_length=2):
    for part in re.split(r'[.,;:()\s\-_?!*]+', value):
        if len(part) >= min_suggest_length:
            yield part


class IDSCache(CacheAdapter):
    """ Базовый класс для кеширования сущностей, которые можно достать из ids
    """
    repo = None  # ids репозиторий, из которого будут браться данные
    fields_field = '_fields'
    limit_field = '_limit'

    def __init__(self, cache, key_field='id', fields=('id', 'name'), filters=None, limit=100):
        self.fields = ','.join(fields)
        self.key_field = key_field
        self.repo_type = self.repo.SERVICE + '.' + self.repo.RESOURCES
        self.filters = filters
        self.limit = limit
        super().__init__(cache, self.repo_type)

    def prepare_cache(self):
        query = {self.fields_field: self.fields, self.limit_field: self.limit}
        if self.filters:
            query.update(**self.filters)
        for obj in self.repo.get(query):
            self.set(obj[self.key_field], obj)

    def get_default(self, short_key):
        lookup = {self.key_field: short_key, self.fields_field: self.fields}
        try:
            return self.repo.get_one(lookup=lookup)
        except BackendError as e:
            if getattr(e, 'response', None) is not None and e.response.status_code == 404:
                log.info('%s does not exists. %s=%s', self.repo_type, self.key_field, short_key)
                return {self.key_field: short_key}
            else:
                log.exception('Cannot fetch %s. %s=%s', self.repo_type, self.key_field, short_key)
        except Exception:
            log.exception('Cannot fetch %s. %s=%s', self.repo_type, self.key_field, short_key)
        return {}


class StaffGroups(IDSCache):
    """ Кеш групп стаффа """
    repo = get_ids_repository('staff', 'group')


class StaffPeople(IDSCache):
    """ Кеш людей стаффа """
    repo = get_ids_repository('staff', 'person')


class Services(IDSCache):
    """ Кеш сервисов abc """
    repo = get_ids_repository('abc', 'service', api_version=4, timeout=5)
    fields_field = 'fields'
    limit_field = 'page_size'


class Metrix(CacheAdapter):
    """
    Кэш просмотров страниц из Яндекс.Метрики
    """

    def __init__(self, cache, source, regex, case_sensitive_key=False):
        """
        :param cache: CacheStorage
        :param source: источник из api.metrix в yaml-конфигах
        :param regex: регулярка для получения ключа из посещённого пути
        :param case_sensitive_key: строгий регистр для ключа
        """
        super().__init__(cache, f'{source}_metrix')
        self.source = source
        self.regex = regex
        self.case_sensitive_key = case_sensitive_key

    def key(self, short_key):
        result = super().key(short_key)
        return result if self.case_sensitive_key else result.lower()

    def get_default(self, short_key):
        return 0

    def prepare_metrix(self):
        try:
            data = get_metrix_data(self.source)
        except Exception as e:
            log.error(e)
            return

        result = Counter()
        regex = re.compile(self.regex)

        for entry in data:
            match = regex.match(entry['path'])
            if match:
                key = match.group('key')
                key = key if self.case_sensitive_key else key.lower()
                result[key] += entry['page_views']

        for key, value in result.items():
            self.set(key, value)


def get_page_links(urls):
    """ Получение ссылок в интранете на текущий урл
    """
    from intranet.search.yt.jobs import extract_links
    result = {
        'texts': '',
        'urls': '',
        'links_count': 0,
    }
    job = extract_links.Job({'env': settings.YENV_TYPE})
    for url in urls:
        data = job.get_links_by_url(url)
        if not data:
            continue
        result['texts'] = '\n'.join([result['texts'], data['texts']])
        result['urls'] = '\n'.join([result['urls'], data['urls']])
        result['links_count'] += data['links_count']
    return result


def get_yt_cache_table(search, index):
    name = '-'.join(['cache', search, index])
    yt_path = os.path.join(settings.ISEARCH['yt']['base_path'], 'data', settings.YENV_TYPE)
    return os.path.join(yt_path, name)


def get_page_clicked_marked_texts(urls):
    """ Получение текстов кликабельных и отмеченных витальными или релевантными
    поисковых запросов для урла.
    """
    from intranet.search.core.models import ClickedMarkedUrl
    data = {
        'queries_clicked': [],
        'queries_vital': [],
        'queries_relevant': [],
    }
    for item in ClickedMarkedUrl.objects.filter(url__in=urls):
        if item.long_clicks_count > 0:
            data['queries_clicked'].append(item.text)
        if item.vital_marks_count > 0:
            data['queries_vital'].append(item.text)
        if item.relevant_marks_count > 0:
            data['queries_relevant'].append(item.text)
    return data


def is_moved_page(url):
    # TODO когда избавимся от FerrymanClient можно будет вынести импорты наверх
    from intranet.search.core.models import MovedPage
    return MovedPage.objects.filter(old_url=url).exists()


def get_page_moved_from(url):
    # TODO когда избавимся от FerrymanClient можно будет вынести импорты наверх
    from intranet.search.core.models import MovedPage
    moved_from = MovedPage.objects.filter(new_url=url).values_list('old_url', flat=True)
    return list(moved_from)


def get_popularity_factors(url):
    """
    Получение факторов, основанных на популярности страницы (отметки на выдаче, клики,
    ссылки из других источников) с учётом перенесённых страниц
    """
    if is_moved_page(url):
        return {'is_moved': True, 'links': {}, 'search_queries': {}}

    moved_from = get_page_moved_from(url)
    all_urls = [url] + moved_from
    result = {
        'is_moved': False,
        'links': get_page_links(all_urls),
        'search_queries': get_page_clicked_marked_texts(all_urls),
    }
    return result
