import os

from django.conf import settings
from django.forms import model_to_dict

from intranet.search.core.models import FerrymanTable
from intranet.search.core.storages import Storage
from intranet.search.core.utils import generate_prefix, parse_prefix


def table_model_to_dict(obj):
    obj_dict = model_to_dict(obj)
    obj_dict['revision'] = model_to_dict(obj.revision)
    return obj_dict


class FerrymanTableStorage(Storage):
    """ Хранилище данных о таблицах для ферримена
    """
    base_path = os.path.join(settings.ISEARCH['yt']['base_path'], 'data', settings.YENV_TYPE)
    push_path = os.path.join(base_path, 'pushes')
    name_sep = '-'

    schema = ['search', 'index', 'revision', 'suffix']

    def create(self, **kwargs):
        t = FerrymanTable.objects.create(**kwargs)
        return table_model_to_dict(t)

    def delete(self, **kwargs):
        FerrymanTable.objects.filter(**kwargs).delete()

    def set_status(self, id_, status):
        FerrymanTable.objects.filter(id=id_).update(status=status)

    def get_active(self):
        statuses = [FerrymanTable.STATUS_FINAL, FerrymanTable.STATUS_ERROR]
        tables = FerrymanTable.objects.exclude(status__in=statuses).select_related('revision')
        return [table_model_to_dict(t) for t in tables]

    def get_finished(self):
        statuses = [FerrymanTable.STATUS_FINAL, FerrymanTable.STATUS_ERROR]
        tables = FerrymanTable.objects.filter(status__in=statuses).select_related('revision')
        return [table_model_to_dict(t) for t in tables]

    def get_all_paths(self):
        return list(FerrymanTable.objects.values_list('path', flat=True))

    def parse_table_path(self, path):
        """ Парсит имя таблицы в YT
        """
        name = path.rsplit('/', 1)[-1]
        data = parse_prefix(self.name_sep, self.schema, name)
        return data

    @classmethod
    def get_yt_table_path(cls, pushes=False, **kwargs):
        """ Возвращает имя таблицы для YT
        """
        path = cls.push_path if pushes else cls.base_path
        name = generate_prefix(cls.name_sep, cls.schema, **kwargs)
        return cls.get_absolute_path(name, path)

    @classmethod
    def get_absolute_path(cls, path, base_path=None):
        base_path = base_path or cls.base_path
        return os.path.join(base_path, path)
