from django.forms import model_to_dict

from intranet.search.core.models import GroupAttr


from .base import ArtefactStorage, BufferedStorageMixin


class GroupAttrStorage(BufferedStorageMixin, ArtefactStorage):
    key_prefix = 'group_attrs:buffer'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.buffer_key = self.revision['id']

    def _create(self, group):
        data = {
            'name': group['name'],
            'value': group['value'],
            'url': group.get('url', ''),
            'label_ru': group.get('label', ''),
            'label_en': group.get('label_en', ''),
        }

        id_ = '{}{}'.format(group['name'], group['value'])
        return id_, data

    def _flush(self, data):
        # если у группировочного атрибуты нет label_ru, label_en, url,
        # то такой атрибут можно в базу не сохранять
        save = [data[name] for name in ('label_ru', 'label_en', 'url')
                if data.get(name)]

        if not save:
            return

        base = self.get_base_lookup()
        base.update({
            'name': data['name'],
            'value': data['value'],
        })

        defaults = {
            'label_ru': data['label_ru'],
            'label_en': data['label_en'],
            'url': data['url'],
        }

        group, created = GroupAttr.objects.get_or_create(defaults=defaults, **base)

        if not created:
            group.label_ru = data['label_ru'] or group.label_ru
            group.label_en = data['label_en'] or group.label_en
            group.url = data['url'] or group.url
            group.save()

    def get(self, **kwargs):
        base = self.get_base_lookup()

        for lookup in ('group', 'id__in', 'value__in', 'search', 'name'):
            if lookup in kwargs:
                base[lookup] = kwargs.pop(lookup)

        queryset = GroupAttr.objects.filter(**base)

        if kwargs.pop('_distinct', False):
            queryset = queryset.distinct()

        fields = kwargs.pop('_fields', [])
        if fields:
            queryset = queryset.values(*fields)

        assert(not kwargs)
        return [model_to_dict(i) for i in queryset] if not fields else list(queryset)

    def purge(self):
        """Удаляет все фасеты для текущей ревизии
        """
        super().purge()
        GroupAttr.objects.filter(**self.get_base_lookup()).delete()

    def count(self, **kwargs):
        base = self.get_base_lookup()
        return GroupAttr.objects.filter(**base).count()
