import datetime
import logging
import json
from itertools import zip_longest
from collections import defaultdict

from django.db.transaction import atomic
from django.forms import model_to_dict
from django.core.cache import caches
from django.db.models import Q, Count

from intranet.search.core import sources
from intranet.search.core.models import Indexation, IndexationStats
from intranet.search.core.storages.revision import RevisionStorage


from .base import Storage, DoesNotExist
from .stage_status import StageStatusStorage


log = logging.getLogger(__name__)


def indexation_model_to_dict(obj):
    return model_to_dict(obj)


def sum_lists(stream1, t1, stream2, t2, window):
    t1 = t1 or 0
    t2 = t2 or 0

    if t1 > t2:
        stream1, stream2 = stream2, stream1
        t1, t2 = t2, t1

    s1 = stream1[-(window - (t2 - t1)):] if window > t2 - t1 else []
    s2 = stream2[-window:]

    result = list(map(sum, zip_longest(s1, s2, fillvalue=0)))
    return result


class IndexationStatsWrapper:
    STATUS_HISTORY_SIZE = 30

    def __init__(self, data=None, tick=0):
        self.data = data or self.get_default()
        self.tick = tick

    def to_json(self):
        return json.dumps(dict(self.data, tick=self.tick))

    def is_empty(self):
        return all(not status['counts'] and not status['total']
                   for value in self.data.values()
                   for status in value.values())

    def __iter__(self):
        for stage, status_info in self.data.items():
            if stage == 'tick':
                # такое случается, если вдруг натыкаемся на старую статистику
                continue

            for status, counts in status_info.items():
                yield (stage, status), counts

    @classmethod
    def get_default(cls):
        return {stage: {status: {'total': 0, 'counts': []}
                        for status in StageStatusStorage.statuses}
                for stage in StageStatusStorage.stages}

    def update(self, stats):
        """Принимает выход функции stage_storage.get_stats()

        Нужно применять аккуратно и апдейтить данные ТОЛЬКО с одного и
        того же хоста иначе могут возникнуть неконсистентности в
        суммарной табличке.

        """
        data = self.data

        for (stage, status), count in stats.items():
            counts_info = data[stage][status]

            counts = counts_info['counts']
            counts.append(count)

            previous_total = counts_info['total']
            if status in StageStatusStorage.terminal_statuses:
                # статусы в терминальных стадиях просто накапливаются
                new_total = previous_total + count
            else:
                # в нетерминыльных, нам интересно лишь мгновенное количество
                new_total = count

            counts_info['total'] = new_total
            counts_info['counts'] = counts[-self.STATUS_HISTORY_SIZE:]

    def merge_with(self, other):
        """ принимает на вход объект того же типа и складывает каунты """
        data = self.data

        for (stage, status), other_counts_info in other:
            counts_info = data[stage][status]

            counts_info['total'] += other_counts_info['total']
            counts_info['counts'] = sum_lists(
                counts_info['counts'], self.tick,
                other_counts_info['counts'], other.tick,
                self.STATUS_HISTORY_SIZE,
            )

        self.tick = max(self.tick, other.tick)

        return self

    def filter(self, stage=None, status=None):
        check = lambda x: hasattr(x, '__contains__')

        if stage is not None and not check(stage):
            stages = [stage]
        else:
            stages = stage

        if status is not None and not check(status):
            statuses = [status]
        else:
            statuses = status

        for (stage, status), counts_info in self:
            if not (stages is None or stage in stages):
                continue

            if statuses is None or status in statuses:
                yield (stage, status), counts_info

    def count(self, status=None, stage=None):
        return sum(counts_info['total'] for _, counts_info in
                    self.filter(stage, status))

    def deltas(self, status=None, stage=None):
        res = [counts_info['counts'] for _, counts_info in
               self.filter(stage, status)]

        return list(map(sum, zip_longest(*res, fillvalue=0)))

    def clear_intermediate_counts(self):
        """ Убирает данные о промежуточных значениям, оставляя только total
        """
        for stage, stage_data in self.data.items():
            for status, stage_status_data in stage_data.items():
                stage_status_data['counts'] = []


class IndexationStorage(Storage):
    def __init__(self):
        super(Storage, self).__init__()
        self.revision_storage = RevisionStorage()

    @property
    def _cache(self):
        return caches['local_redis']

    def is_stage_allowed(self, id_, stage_name):
        """Проверяем доступна ли стадия у индексации
        """
        try:
            return self.get_cached(id_).get(stage_name, True)
        except DoesNotExist:
            return False

    def get_status(self, id_):
        """Возвращает статус индексации
        """
        return self.get(id_)['status']

    def create(self, revision, comment, user):
        """Создает новую индексацию
        """
        indexation = Indexation.objects.create(search=revision['search'],
                                               index=revision['index'],
                                               backend=revision['backend'],
                                               revision_id=revision['id'],
                                               start_time=datetime.datetime.now(),
                                               status='new',
                                               comment=comment,
                                               user=user)
        log.info('Indexation created: %s, %s', indexation.id, model_to_dict(indexation))
        return indexation.id

    def get(self, id_):
        try:
            obj = Indexation.objects.get(pk=id_)
            res = indexation_model_to_dict(obj)
        except Indexation.DoesNotExist:
            raise DoesNotExist('Indexation does not exist: %s' % id_)

        return res

    def exists(self, id_):
        return Indexation.objects.filter(pk=id_).exists()

    def get_by_id_list(self, id_list):
        qs = Indexation.objects.filter(id__in=id_list)
        return [indexation_model_to_dict(i) for i in qs]

    def get_cached(self, id_):
        key = 'indexation_cache' + str(id_)
        res = self._cache.get(key)

        if res is None:
            res = self.get(id_)
            self._cache.set(key, res)

        return res

    def purge(self, id_):
        """Помечает индексацию как неактуальную
        """
        pass  # ничего не  делаем. сохраняем информацию для истории

    @atomic
    def update(self, id_, **kwargs):
        """Обновляет поля индексации
        """
        log.debug('Indexation updated: %s, %s', id_, kwargs)
        msg = kwargs.pop('msg', None)
        if msg:
            old_msg = self.get(id_)['log']
            kwargs['log'] = (old_msg or '') + (f'\n{datetime.datetime.now()} - {msg}')

        Indexation.objects.filter(id=id_).update(**kwargs)
        if kwargs.get('status') == 'done' and kwargs.get('end_time'):
            data = {'latest_indexation_time': kwargs['end_time']}
            filters = (
                Q(indexation=id_) & (
                    Q(latest_indexation_time__lt=kwargs['end_time']) |
                    Q(latest_indexation_time__isnull=True)
                )
            )

            self.revision_storage.update(filters, data=data)

    @atomic
    def update_wiki_since(self, id_, new_since):
        log.debug('Setting wiki since value: %s, for id: %s', new_since, id_)
        Indexation.objects.filter(id=id_).update(
            last_wiki_since=new_since,
        )

    def set_indexer(self, id_, indexer):
        """Сериализует индексатор и сохраняет его в индексации
        """
        self.update(id_, options=indexer.options)
        log.info('Set indexation indexer: %s, %s', id_, indexer.options)

    def get_indexer(self, id_=None, indexation=None):
        """Возвращает объект индексатора для данной индексации
        """
        if not indexation:
            indexation = self.get(id_)
        options = indexation['options']
        if options is None:
            log.warning('Got indexation without options %s', id_)
            options = {'revision': self.revision_storage.get(indexation['revision']),
                       'indexation_id': indexation['id']}

        try:
            Source = sources.load(indexation['search'], indexation['index'])
            indexer = Source(options)
        except Exception:
            log.warning('Cannot load indexer for indexation: %s', id_)
            return None

        return indexer

    def get_last(self, amount=10, **kwargs):
        lookup = {}
        for key in ('search', 'index', 'revision', 'status', 'status__in'):
            if key in kwargs:
                lookup[key] = kwargs.pop(key)
        assert not kwargs
        indexations = Indexation.objects.filter(**lookup).order_by('-start_time')
        indexations = indexations[:amount]
        return [indexation_model_to_dict(i) for i in indexations]

    def get_latest_indexation(self, search, index, revision, status__in=['done', 'new']):
        result = self.get_last(amount=1,
                               search=search,
                               index=index,
                               revision=revision,
                               status__in=status__in)

        if result:
            indexation = result[0]
            indexation['indexer'] = self.get_indexer(indexation['id'])
            return indexation

    def count(self, **kwargs):
        lookup = {}
        for key in ('search', 'index', 'revision'):
            if key in kwargs:
                lookup[key] = kwargs.pop(key)
        assert not kwargs
        return Indexation.objects.filter(**lookup).count()

    def touch(self, id_):
        """Выставляет last_check_at в текущий момент
        """
        self.update(id_, last_check_at=datetime.datetime.now())

    def get_with_stale_check(self, seconds):
        """Возвращает id индексаций, для которых за последние seconds секунд
        не запускался check
        """
        threshold = datetime.datetime.now() - datetime.timedelta(seconds=seconds)
        date_filter = Q(last_check_at__lt=threshold) | Q(last_check_at__isnull=True)
        indexations = (
            Indexation.objects
            .filter(date_filter, status='new')
            .values_list('id', flat=True)
        )
        return indexations

    def get_running(self, **kwargs):
        """ Возвращает запущенные в данные момент индексации
        """
        lookup = {
            'status__in': ('new', 'stop'),
        }
        lookup.update(kwargs)

        res = Indexation.objects.filter(**lookup)
        return [indexation_model_to_dict(i) for i in res]

    @atomic
    def update_host_stage_stats(self, indexation_id, revision_id, hostname, stats=None):
        """ Обновляет статистику для заданной индекции на заданном хосте
        """
        old_stats, created = IndexationStats.objects.get_or_create(
            revision_id=revision_id, indexation_id=indexation_id, hostname=hostname)

        if created:
            cur_stats = IndexationStatsWrapper()
        else:
            cur_stats = IndexationStatsWrapper(old_stats.stats, old_stats.tick)

        if stats:
            cur_stats.update(stats)

        old_stats.stats = cur_stats.data
        old_stats.tick += 1
        old_stats.save()

    @atomic
    def archive_stage_stats(self, indexation_id, revision_id):
        """ Архивирует статистику индексаций: собирает данные по всем хостам, оставляет только
        total значения и сохраняет это в виде одной записи в IndexationStats
        """
        stats = self.get_stage_stats(indexation_id)
        stats.clear_intermediate_counts()

        IndexationStats.objects.filter(indexation_id=indexation_id).delete()
        IndexationStats.objects.create(
            revision_id=revision_id, indexation_id=indexation_id, hostname='',
            tick=stats.tick, stats=stats.data
        )

    def get_stage_stats(self, indexation_id, revision_id=None):
        """ Возвращает враппер с объединенной статистикой по всем хостам
        """
        query = IndexationStats.objects.filter(indexation_id=indexation_id)
        if revision_id:
            query = query.filter(revision_id=revision_id)

        res = IndexationStatsWrapper()
        for stat in query:
            res.merge_with(IndexationStatsWrapper(stat.stats, stat.tick))

        return res

    def get_combined_stage_stats(self, indexation_ids):
        res = IndexationStatsWrapper()

        for indexation_id in indexation_ids:
            res.merge_with(self.get_stage_stats(indexation_id))
        return res

    def get_hosts(self, indexation_id, revision_id=None, with_date=False, **kwargs):
        """ Возвращает список хостов, где есть статистика для данной индаксации
        """
        if not with_date:
            values_args = ('hostname',)
            values_kwargs = {'flat': True}
        else:
            values_args = ('hostname', 'updated_at')
            values_kwargs = {}

        query = IndexationStats.objects.filter(indexation_id=indexation_id, **kwargs)
        if revision_id:
            query = query.filter(revision_id=revision_id)

        return list(query.order_by('id').values_list(*values_args, **values_kwargs))

    def get_combined_hosts(self, indexation_ids, **kwargs):
        res = []
        for indexation_id in indexation_ids:
            res.extend(self.get_hosts(indexation_id, **kwargs))
        return res

    def get_by_status_count(self, **kwargs):
        query = (
            Indexation.objects
            .values('status', 'search', 'index')
            .annotate(count=Count('id'))
        )
        if kwargs:
            query = query.filter(**kwargs)

        result = defaultdict(dict)
        for obj in query:
            result[obj['status']][(obj['search'], obj['index'])] = obj['count']

        return result

    def delete_older_than(self, old_threshold, interval=100000):
        idx_qs = Indexation.objects.exclude(status=Indexation.STATUS_NEW)
        stats_qs = IndexationStats.objects.exclude(indexation__status=Indexation.STATUS_NEW)

        first_indexation = idx_qs.filter(start_time__lte=old_threshold).first()
        last_indexation = idx_qs.filter(start_time__lte=old_threshold).last()

        if not first_indexation:
            return

        from_id = first_indexation.id
        last_id = last_indexation.id

        while from_id < last_id:
            till_id = min(from_id + interval, last_id)

            log.info('Delete IndexationStats from %s to %s', from_id, till_id)
            qs = stats_qs.filter(indexation__id__gte=from_id,
                                 indexation__id__lte=till_id)
            qs._raw_delete(qs.db)

            log.info('Delete Indexations from %s to %s', from_id, till_id)
            qs = idx_qs.filter(id__gte=from_id, id__lte=till_id)
            qs._raw_delete(qs.db)

            from_id = till_id
