from django.conf import settings
from intranet.search.core import mongodb
from intranet.search.core.storages import Storage


class ProfileStorage(Storage):
    def __init__(self):
        self.collection_name = 'profiles'

    def create(self, entity, key, attributes):
        """Создает запись с атрибутами сущности `entity` по ключу `key`
        attributes = {'attribute_name': 'attribute_value', ...}

        """
        assert isinstance(attributes, dict)

        attributes['_id'] = key
        self._get_collection(entity).insert(attributes)

    def update(self, entity, key, attributes):
        """Обновить данные. Метод вызывается после завершения джоба
        для кажой строчки выходной таблицы

        attributes = {'attribute_name': 'attribute_value', ...}

        """
        self._get_collection(entity).update_one(
            {'_id': key}, {'$set': attributes}, upsert=True)

    def get(self, entity, key, attribute_names=None):
        """ Получение значений атрибутов по ключу `key` в сущности `entity` """
        res = self._get_collection(entity).find_one(
            {'_id': key}, attribute_names or None)
        return {} if res is None else res

    def get_avg(self, entity, attribute_name):
        """ Среднее значение аттрибута """
        res = self._get_collection(entity).aggregate(
            {'$group': {'_id': None, 'mean': {'$avg': '$' + attribute_name}}})

        if res['ok']:
            res = res['result'][0]['mean']
        else:
            res = None

        return res

    def list_all(self, entity, from_='', per_page=100):
        return self._get_collection(entity).find(
            {'_id': {'$gt': from_}}
        ).sort('_id').limit(per_page)

    def count(self, entity):
        return self._get_collection(entity).count()

    def _get_collection(self, entity):
        return mongodb.client[settings.ISEARCH_MONGODB_DB][
            self.collection_name + '_' + entity]
