import math

from intranet.search.core.swarm import Indexer
from intranet.search.core.utils import reraise_as_recoverable, http


class PagedApiIndexer(Indexer):
    """ Базовый класс индексаторов с использованием апи сервиса, использующих постраничную пагинацию.
    """
    snippet_languages = ('ru', 'en')

    def fetch_objects(self, page=1):
        """ Извлекает объекты для индексации

        :return: iterable
        """
        raise NotImplementedError()

    def get_counts(self):
        """ Возвращает количество страниц и количество документов

        :return: tuple, первый элемент - количество страниц, второй количество документов
        """
        return None, None

    @property
    def need_fetch(self):
        return hasattr(self, 'do_fetch')

    def do_setup(self, **kwargs):
        walk_chunks = self.options['threads']
        pages_count, docs_count = self.get_counts()
        if docs_count is not None:
            self.indexation_storage.update(self.indexation_id, expected_documents_count=docs_count)

        if pages_count:
            frame_size = int(math.ceil(float(pages_count) / walk_chunks))

            for i in range(0, walk_chunks):
                page = frame_size * i + 1
                if page > pages_count:
                    break
                last_page = min(frame_size * (i + 1), pages_count)
                self.next('walk', page=page, last_page=last_page)
        else:
            self.next('walk')

    @reraise_as_recoverable(*http.ERRORS)
    def do_walk(self, page=1, last_page=0):
        objects = self.fetch_objects(page)

        for obj in objects:
            func = 'fetch' if self.need_fetch else 'create'
            self.next(func, obj=obj)

        if page < last_page:
            self.next('walk', page=page + 1, last_page=last_page)

    def do_create(self, obj, **kwargs):
        doc = self.create_document(self.get_doc_url(obj), updated=self.get_doc_updated(obj))
        doc.emit_body(self.create_body(obj, **kwargs))
        self.emit_factors(doc, obj)
        self.emit_attrs(doc, obj)

        for lang in self.snippet_languages:
            snippet = self.create_snippet(obj, lang, **kwargs)
            doc.emit_snippet(snippet, lang)

        self.next('store', document=doc)

    def do_push(self, data, delete=False):
        if delete:
            self.do_delete(data)
        else:
            func = 'fetch' if self.need_fetch else 'create'
            self.next(func, obj=data)

    def do_delete(self, obj):
        doc = self.create_document(self.get_doc_url(obj), updated=self.get_doc_updated(obj))
        self.next('store', document=doc, delete=True)

    def get_doc_url(self, obj):
        """ Возвращает url документа для индексации
        """
        raise NotImplementedError()

    def get_doc_updated(self, obj):
        """ Возвращает дату изменения объекта или None, если её нет
        """
        return None

    def create_body(self, obj, **kwargs):
        raise NotImplementedError()

    def create_snippet(self, obj, lang='ru', **kwargs):
        raise NotImplementedError()

    def emit_factors(self, doc, obj):
        pass

    def emit_attrs(self, doc, obj):
        pass
