from logging import getLogger

from celery import Task, shared_task
from django import db
from requests import RequestException
from ylog.context import log_context

from intranet.search.core import storages
from intranet.search.core.sources.directory import client
from intranet.search.core.swarm.pushes import handle_indexation_push, handle_single_push, kill_push

logger = getLogger(__name__)


class BaseDirectoryTask(Task):
    abstract = True
    default_retry_delay = 5
    max_retries = 5
    autoretry_for = (db.Error, RequestException)

    def __call__(self, organization, *args, **kwargs):
        """ Автоматически ретраим таски в случае некоторых исключений
        """
        with log_context(org_directory_id=organization['directory_id'], org_id=organization['id'],
                         event=kwargs.get('event'), push_id=kwargs.get('push_id')):
            try:
                return super().__call__(organization, *args, **kwargs)
            except self.autoretry_for as e:
                logger.warning('Got error: %s. Trying to retry. Retries attempt: %s',
                               e, self.request.retries)
                self.retry(countdown=self.default_retry_delay ** self.request.retries)


@shared_task(name='isearch.tasks.sync_directory_organization', base=BaseDirectoryTask)
def sync_directory_organization(organization, force_reindex=False, push_id=None, **kwargs):
    """ Синхронизация данных директории с директорией

    :param organization: данные нашей организации
    :param force_reindex: принудительно переиндексировать организацию
    :param kwargs: данные события
    """
    if kwargs.get('revision') and kwargs['revision'] < organization['directory_revision']:
        # ничего не делаем, если у нас уже есть более новые данные
        logger.warning('Got directory data older than we have: data=%s', kwargs)
        kill_push(push_id, cancel=True, comment='Got directory data older than we have')
        return

    org_repo = storages.OrganizationStorage()
    dir_client = client.DirectoryApiClient()
    try:
        dir_data = dir_client.get_organization(organization['directory_id'])
    except client.ObjectDoesNotExist:
        logger.warning('Organization does not exist: directory_id=%s', organization['directory_id'])
        org_repo.delete(organization['id'])
        kill_push(push_id, cancel=True, comment='Organization does not exist')
    else:
        if organization.get('deleted'):
            logger.info('Organization restored: directory_id=%s', organization['directory_id'])
            org_repo.restore(organization['id'])

        organization = org_repo.update_from_directory_data(dir_data)
        logger.info('Organization updated from directory: directory_id=%s', organization['directory_id'])
        if force_reindex:
            # для новой организации сразу запускаем индексацию директории
            comment = 'Force reindex. event={}, org_directory_id={}.'.format(
                kwargs.get('event'), organization['directory_id'])
            for index in ('', 'groups', 'departments'):
                handle_indexation_push(push_id, search='directory', index=index,
                                       comment=comment, organization_id=organization['id'])
                logger.info('Directory push handled: index=%s, push_id=%s', index, push_id)
        else:
            kill_push(push_id, cancel=True, comment='Nothing to do by push')


@shared_task(name='isearch.tasks.sync_directory_object', base=BaseDirectoryTask)
def sync_directory_object(organization, index='', action='create', push_id=None, **kwargs):
    """ Синхронизация данных об одном объекте из директории (пользователе, группе, департаменте)
    :param organization: словарь данных о нашей организации
    :param index: поисковый индекс объекта
    :param kwargs: данные события
    """
    handle_single_push(push_id, search='directory', index=index, data=kwargs['object'],
                       action=action, organization_id=organization['id'])


@shared_task(name='isearch.tasks.sync_department_group_members', base=BaseDirectoryTask)
def sync_department_group_members(organization, type='group', push_id=None, **kwargs):
    """ Синхронизация пользователей департамента или группы
    """
    comment = 'event={}, {}={}, org_directory_id={}'.format(
        kwargs['event'], type, kwargs['object']['id'], organization['directory_id'])
    keys = ['{}:{}'.format(type, kwargs['object']['id'])]

    handle_indexation_push(push_id, search='directory', index='', comment=comment,
                           organization_id=organization['id'], keys=keys)
