import socket
import logging

from datetime import timedelta
from uuid import uuid4

from celery import shared_task
from ylog.context import log_context

from intranet.search.core.utils import get_current_node, get_active_hosts
from intranet.search.core import storages


log = logging.getLogger(__name__)


@shared_task(name='isearch.tasks.heartbeat')
def heartbeat():
    node = get_current_node(create_missing=True)
    node.save()


@shared_task(name='isearch.tasks.collect_indexation_stats')
def collect_indexation_stats(revision_id, indexation_id, with_global=False):
    """ для текущей ноды получает статистику из локального редиса
        и записывает в базу

    если with_global == True то собрать еще данные из базы

    """
    with log_context(indexation_id=indexation_id, collect_id=uuid4().hex,
                     method='collect_indexation_stats'):
        stage_storage = storages.LocalStageStatusStorage(revision_id, indexation_id)
        indexation_storage = storages.IndexationStorage()

        if not indexation_storage.exists(indexation_id):
            log.warning('Collect stats, but indexation does not exists: %s', indexation_id)
            stage_storage.delete()
            return

        cur_host = socket.gethostname()

        log.debug('Collect stats for host: %s, with_global: %s', cur_host, with_global)
        stats = stage_storage.get_stats(with_global)

        if stats:
            log.debug('Stats %s %s: %s', indexation_id, cur_host, stats)
            indexation_storage.update_host_stage_stats(indexation_id, revision_id, cur_host, stats)


@shared_task(name='isearch.tasks.collect_stats')
def collect_stats(indexation_id=None, sync=False):
    """ Запускает сбор статистик стадий всех запущенных индексаций
    или если передан indexation_id то только для этой индексации
    """
    indexation_storage = storages.IndexationStorage()
    cur_host = socket.gethostname()

    if sync:
        with_global = True
    else:
        active_nodes = get_active_hosts(timedelta(minutes=10))
        with_global = active_nodes[0] == cur_host if active_nodes else False

    log.info('Starts collect_stats on host: %s, with_global: %s', cur_host, with_global)

    if indexation_id is None:
        # Выбирать только те индексации и ревизии, для которых есть статистика в базе
        rev_ind_pairs = storages.LocalStageStatusStorage.get_indexations(with_global=with_global,
                                                                         with_global_limit=1000)
    else:
        indexation = indexation_storage.get(indexation_id)
        rev_ind_pairs = [(indexation['revision'], indexation['id'])]

    for revision_id, indexation_id in rev_ind_pairs:
        log.info('Running stats collection for indexation %s, revision %s, host %s, with_global %s',
                 indexation_id, revision_id, cur_host, with_global)
        collect_indexation_stats(revision_id, indexation_id, with_global=with_global)


@shared_task(name='isearch.tasks.delete_finished_stats')
def delete_finished_stats():
    """ Собираем статистику отмененных и завершенных индексаций
    про которые осталсь информация в локальном хранилище
    """
    indexation_storage = storages.IndexationStorage()
    ids = {(r_id, i_id) for r_id, i_id in storages.LocalStageStatusStorage.get_indexations()}
    running_ids = {(i['revision'], i['id']) for i in indexation_storage.get_running()}

    finished_ids = ids - running_ids
    for revision_id, indexation_id in finished_ids:
        storages.LocalStageStatusStorage(revision_id, indexation_id).delete()
