from django.utils.encoding import smart_str

from .http import normalize_url


class CacheAdapter:
    def __init__(self, cache, name):
        self.cache = cache
        self.name = name

    def __getitem__(self, short_key):
        data = self.get(short_key)
        if not data:
            default = self.get_default(short_key)
            self.set(short_key, default)
            return default
        return data

    def get_default(self, short_key):
        return {}

    def get(self, short_key, default=None):
        return self.cache.get(self.key(short_key), default)

    def set(self, short_key, data):
        self.cache.set(self.key(short_key), data)

    def key(self, short_key):
        return f'{smart_str(self.name)}_{smart_str(short_key)}'


class Urls(CacheAdapter):
    def __init__(self, cache, name='doc_urls'):
        super().__init__(cache, name)
        # локальный кэш для случаев когда
        # с одной страницы очень много ссылок на другую
        self._visited_cache = set()

    def _make_key(self, url):
        return normalize_url(url)

    def is_url_visited(self, url):
        if url is None:
            return False

        url = self._make_key(url)
        if url in self._visited_cache:
            return True

        res = self.get(url) or False
        if res:
            self._visited_cache.add(url)

        return res

    def visit_url(self, url):
        if url is None:
            return False

        url = self._make_key(url)
        visited = self.is_url_visited(url)

        if not visited:
            self._visited_cache.add(url)
            self.set(url, True)

        return (not visited)
