import re
import time
import socket
import logging
from urllib.parse import parse_qsl, urlsplit, urlunsplit, urljoin, urlencode, quote, unquote

import requests
from ids.exceptions import BackendError
from requests import HTTPError, codes
from requests.adapters import HTTPAdapter
from requests.exceptions import ChunkedEncodingError
from requests.utils import requote_uri

requests.packages.urllib3.disable_warnings()

log = logging.getLogger(__name__)

MAX_RETRIES = 3

ERRORS = (
    HTTPError,
    requests.ConnectionError,
    requests.Timeout,
    socket.error,
    BackendError,
    # в эту ошибку requests оборачивает Connection reset by peer
    ChunkedEncodingError,
)

try:
    from OpenSSL.SSL import SysCallError, Error
    ERRORS += (SysCallError, Error)
except ImportError:
    pass


# Кешируем все сессии тут, чтобы переиспользовать коннеты к внешним источникам
__sessions = {}


def _get_key(obj):
    return str(sorted(obj.items()))


class ProtectedSession(requests.Session):
    __attrs__ = requests.Session.__attrs__ + ['_write_protected']

    def __init__(self):
        super().__setattr__('_write_protected', False)
        super().__init__()

    def __setattr__(self, name, value):
        if self._write_protected:
            raise AttributeError("You can't change attributes of ProtectedSession")
        else:
            super().__setattr__(name, value)

    def protect(self):
        self._write_protected = True

    def close(self):
        """ Не закрываем никакие адаптеры """
        pass


def create_session(**kwargs):
    key = _get_key(kwargs)
    session = __sessions.get(key)

    if session is not None:
        return session

    session = ProtectedSession()

    adapter_kw = {'pool_connections': 100, 'pool_maxsize': 100}

    if 'max_retries' in kwargs:
        adapter_kw['max_retries'] = kwargs.pop('max_retries')

    adapter = HTTPAdapter(**adapter_kw)
    session.mount('http://', adapter)
    session.mount('https://', adapter)

    for k, v in kwargs.items():
        if k in session.__attrs__:
            setattr(session, k, v)
        else:
            raise AttributeError('Unknown session attribute %s' % k)

    session.protect()
    __sessions[key] = session
    return session


def call_with_retry(session_func, url, *args, **kwargs):
    """ ok_statuses - статусы на которые не рейзится ошибка
    и не делаются ретраи.
    """
    # TODO: Можно заменить на стандартные ретраи из requests
    retries = kwargs.pop('_retries', MAX_RETRIES)
    ok_statuses = kwargs.pop('_ok_statuses', (codes.ok,))

    while True:
        try:
            response = session_func(url, *args, **kwargs)

            if response.status_code not in ok_statuses:
                response.raise_for_status()

        except ERRORS:
            retries -= 1

            if retries < 0:
                log.exception('Error while request %s', url)
                raise

            time.sleep(2 ** (MAX_RETRIES - retries))
        else:
            return response


def normalize_url(url):
    """
    Normalize given url

    * Lowercase for host
    * Percent-encoding for path
    * Fragments (part after #, '#smth') will be cutted
    * If path is empty, '/' will be added to host
    * sort query params
    * add http if no scheme
    """
    (scheme, netloc, path, query, fragment) = urlsplit(url)

    if not scheme:
        scheme = 'http'

    netloc = netloc.lower()

    path = quote(unquote(path))
    if not path:
        path = '/'
    path = re.sub(r'/+', '/', path)

    query = parse_qsl(query, keep_blank_values=True)
    query.sort()
    query = urlencode(query)
    return requote_uri(urlunsplit((scheme, netloc, path, query, None)))


def absolute_url(url, base_url):
    if url.startswith('http://') or url.startswith('https://'):
        return url

    return urljoin(base_url, url)


SCHEME_RE = re.compile('https?://')


def cut_sheme(url):
    return SCHEME_RE.sub('', url)


def cut_query(url):
    (scheme, netloc, path, query, fragment) = urlsplit(url)
    return urlunsplit((scheme, netloc, path, None, None))
