import json
from collections import OrderedDict
from logging import getLogger

import requests
from django.conf import settings

from intranet.search.abovemeta.search_response import SaaSSearchResponse
from intranet.search.abovemeta.search_settings import sources_for_meta
from intranet.search.core.storages import RevisionStorage
from intranet.search.core.tvm import tvm2_client
from intranet.search.core.utils import http, get_kps

log = getLogger(__name__)

FACTOR_KEYS = ['static_factors', 'zone_factors', 'user_factors', 'dynamic_factors']


def get_saas_config(service):
    endpoint = settings.ISEARCH['api']['saas_admin']
    query = {'path': '/configs/{service}/relev.conf-{service}'.format(service=service)}
    res = requests.get(endpoint.url(query=query), verify=settings.ISEARCH_CA_CERTS)
    res.raise_for_status()
    data = res.json()
    return json.loads(data['storage']['filecontent'], object_pairs_hook=OrderedDict)


def get_all_factors(service: str) -> dict[str, int]:
    """ Возвращает словарь всех факторов где ключ - название фактора, значение - его id в саасе
    """
    conf = get_saas_config(service)
    factors = {}

    for factor_type in FACTOR_KEYS:
        for name, value in conf.get(factor_type, {}).items():
            saas_id = value['index'] if isinstance(value, dict) else value
            factors[name] = saas_id

    return factors


def get_factor_names(service: str) -> list[str]:
    """ Достает имена всех наших факторов из настроек сааса.
    Собирает их в массив, где индекс фактора равен id фактора в конфиге сааса.
    На месте пропусков вставляет факторы "factor_<index>". Это приходится делать, потому что
    иначе формулу невозможно использовать в SaaS из-за неправильных факторов.

    Например, если в саасе такие факторы: {'STAT_meta_obsolete': 1, 'STAT_meta_startpage': 3}
    То здесь вернется: ['factor_0', 'STAT_meta_obsolete', 'factor_2', 'STAT_meta_startpage']
    """
    factors = get_all_factors(service)
    factors_dict = {v: k for k, v in factors.items()}
    factor_names = []
    for f_index in range(0, max(factors_dict) + 1):
        factor_names.append(factors_dict.get(f_index) or f'factor_{f_index}')
    return factor_names


def get_doc_count(revision, text):
    endpoint = settings.ISEARCH['api']['saas'][revision['service']]['search']

    query = {
        'text': text,
        'haha': 'da',
        'relev': 'attr_limit=10000000',
        'timeout': 99999999,
        'pron': 'noprune',
        'ms': 'proto',
        'msp': 'no',
        'hr': 'json',
        'robot': 'da',
        'kps': get_kps(revision['id'])
    }

    url = endpoint.url(query=query)
    headers = {settings.TVM2_SERVICE_HEADER: tvm2_client.get_service_ticket('saas')}

    try:
        session = http.create_session()
        resp = http.call_with_retry(session.get, url, timeout=10, headers=headers)
        count = resp.json()['TotalDocCount'][0]
    except Exception:
        log.exception('Cannot get doccount from saas, revision=%s', revision['id'])
        count = None

    return {'count': count, 'url': url}


def get_indextstat_doc_count(revision):
    """ Количество документов по ревизии, на основе статистики SaaS
    """
    endpoint = settings.ISEARCH['api']['saas_service_stat']['indexstat']
    url = endpoint.url().format(service=revision['service'])
    headers = {settings.TVM2_SERVICE_HEADER: tvm2_client.get_service_ticket('saas')}
    session = http.create_session()
    resp = http.call_with_retry(session.get, url, timeout=10, headers=headers)
    if resp.ok:
        stat = resp.json()
        count = stat['KPS'].get(str(get_kps(revision['id'])))
    else:
        count = None

    return {'count': count, 'url': url}


def get_revisions_for_meta():
    sources = set(sources_for_meta('meta_with_at'))
    revision_storage = RevisionStorage()
    all_active = revision_storage.get_all_active()
    revisions = [rev for rev in all_active if (rev['search'], rev['index']) in sources]
    return revisions


def get_metasearch_factors_for_url(query, url):
    revisions = get_revisions_for_meta()
    return get_factors_for_url(query, url, revisions)


def get_factors_for_url(querytext, url, revisions, **extra):
    """Получить факторы по паре запрос+url"""

    service = revisions[0]['service']
    kpss = ','.join(str(get_kps(r['id'])) for r in revisions)

    # Примерный урл такой:
    # http://saas-searchproxy.yandex.net:17000/?spcctx=doc&service=intrasearch-wiki&text=%D0%B0%D1%80%D0%BA%D0%B0%D0%B4%D0%B8%D1%8F&timeout=99999999&pron=noprune&ms=proto&msp=no&hr=da&robot=da&kps=77275%2C78023%2C78021%2C78022%2C76917&fsgta=_JsonFactors&relev=all_factors&format=json&dbgrlv=da&au=http%3A%2F%2Fclubs.at.yandex-team.ru%2Farcadia%2F24169'

    endpoint = settings.ISEARCH['api']['saas'][service]['search']

    query = {
        'text': querytext,
        'timeout': 99999999,
        'pron': 'noprune',
        'ms': 'proto',
        'msp': 'no',
        'hr': 'da',
        'robot': 'da',
        'kps': kpss,
        'fsgta': '_JsonFactors',
        'relev': 'all_factors',
        'format': 'json',
        'dbgrlv': 'da',
        'au': url,
    }
    query.update(extra)

    saas_url = endpoint.url(query=query)
    headers = {settings.TVM2_SERVICE_HEADER: tvm2_client.get_service_ticket('saas')}
    try:
        session = http.create_session()
        resp = http.call_with_retry(session.get, saas_url, timeout=10, headers=headers)
        response_data = resp.json()
    except Exception:
        log.exception('Cannot get doccount from saas, revisions=%s', kpss)
        return None
    data = SaaSSearchResponse(response_data)
    docs = list(data.get_docs())
    if not docs:
        return None
    doc = docs[0]
    factors = doc.factors
    return factors
