import fnmatch
import hashlib
import os


def _get_env_dicts(prefix):
    """Ищет переменные окружения по заданному префиксу. В найденных откусывает перфикс, приводит хвост
    к нижнему регистру и возврщается его как имя.

    Значение переменной парсится в словарь. Например:
        'queue:store,content;concurrency:10;app:local;pool:threads'
        ->
        {'app': 'local',
         'concurrency': '10',
         'pool': 'threads',
         'queue': 'store,content'}
    """
    for name in fnmatch.filter(list(os.environ), '%s*' % prefix):
        value = os.getenv(name).strip()
        data = dict(kv.strip().split(':') for kv in value.split(';')) if value else {}
        name = name.split(prefix)[1].lower()

        yield name, data


def get_queues():
    # TODO: это должно быть можно объединить с get_celery_workers_config
    for name, data in _get_env_dicts('ISEARCH_CELERY_WORKER_'):
        partitions = data.get('partitions', '1')
        for queue in data['queue'].split(','):
            yield queue, partitions


def get_celery_workers_config():
    defaults = {
        'queue': None,
        'pool': 'prefork',
        'concurrency': '1',
        'app': 'global',
        'partitions': None,
        'suffix': '',
        'loglevel': 'warning',
    }

    for name, data in _get_env_dicts('ISEARCH_CELERY_WORKER_'):
        data = dict(defaults, **data)

        if data['suffix'] and data['partitions']:
            raise RuntimeError('Suffix and partition count cannot be set simultaneously')

        if data['partitions']:
            partitions = int(data['partitions'])

            if partitions == 1:
                data['suffix'] = ''
            else:
                if os.environ.get('DEPLOY_STAGE_ID'):
                    pod_id = os.environ.get('DEPLOY_POD_ID')
                    pod_hash = int(hashlib.sha256(pod_id.encode('utf-8')).hexdigest(), 16)
                    data['suffix'] = str(pod_hash % partitions)
                else:
                    instance_num = get_instance_num()
                    data['suffix'] = str(instance_num % partitions)

        if data['queue']:
            data['queue'] = data['queue'].split(',')

            if data['suffix']:
                # задать суффикс (либо партицировать) можно только для walk или load очередей
                if not set(data['queue']) <= {'walk', 'load'}:
                    raise RuntimeError('Suffix for queues in `%s` celery worker not allowed' % name)

                data['queue'] = [q + data['suffix'] for q in data['queue']]

        data['name'] = name

        yield name, data


def get_celery_beats_config():
    defaults = {
        'app': 'global',
        'loglevel': 'warning',
    }

    for name, data in _get_env_dicts('ISEARCH_CELERY_BEAT_'):
        data = dict(defaults, **data)

        data['name'] = name

        yield name, data


def get_instance_num():
    if 'QLOUD_COMPONENT' not in os.environ:
        return 1

    component = os.environ['QLOUD_COMPONENT']
    instance = os.environ['QLOUD_INSTANCE']
    return int(instance.replace(component, ''))
