import yaml

import yenv
import os
from library.python import resource

from .merge import merge
from .path import resolve_one


def load_path(key):
    return yaml.unsafe_load(resource.resfs_read(key))


def load(path, *overrides, **kwargs):
    config = load_path(path)

    for override in overrides:
        override_config = load_path(override)
        config = merge(config, override_config)

    process(config, **kwargs)

    return config


def process(data, frame=None, **kwargs):
    if frame is None:
        frame = data

    if '$extends' in frame:
        v = frame.pop('$extends')

        assert(v.startswith('~'))

        base = resolve_one(v[1:], data)

        process(data, base, **kwargs)

        derived = frame.copy()
        frame.clear()
        frame.update(merge(base, derived))

    for k, v in frame.items():
        if isinstance(v, dict):
            result = process(data, v, **kwargs)

            if result is not None:
                frame[k] = result
        elif isinstance(v, list):
            result = []
            for v_piece in v:
                if isinstance(v_piece, dict):
                    process(data, v_piece, **kwargs)
                result.append(v_piece)
            frame[k] = result
        else:
            if isinstance(v, str) and v.startswith('~'):
                frame[k] = resolve_one(v[1:], data)

    if '$env' in frame or '$env.type' in frame:
        try:
            env = frame.pop('$env')
        except KeyError:
            env = frame.pop('$env.type')

        try:
            result = merge(frame, yenv.choose_key_by_type(env))
        except ValueError:
            pass
        else:
            frame.clear()
            frame.update(result)

    if '$deploy' in frame:
        deploy_subframe = frame.pop('$deploy')
        in_deploy = 'DEPLOY_BOX_ID' in os.environ
        if in_deploy:
            try:
                result = merge(frame, deploy_subframe)
            except ValueError:
                pass
            else:
                frame.clear()
                frame.update(result)

    if '$env.name' in frame:
        env = frame.pop('$env.name')

        try:
            result = merge(frame, yenv.choose_key_by_name(env))
        except ValueError:
            pass
        else:
            frame.clear()
            frame.update(result)

    klass = frame.get('$class', None)

    if klass:
        if isinstance(klass, str):
            klass = kwargs['classes'][klass]

        return klass(**frame)
