import copy
from urllib.parse import urlunsplit

from django.utils.datastructures import MultiValueDict

from intranet.search.settings import utils
from intranet.search.core.utils import get_oauth_header, get_oauth_token, get_x_api_key_header


class UrlBuilder:

    def __init__(self, endpoint, priority_params=None):
        if priority_params is None:
            priority_params = {}

        params = {}
        params.update(endpoint)
        params.update(priority_params)

        self.schema = params.get('schema', 'http')
        self.host = params.get('host', 'localhost')
        self.path = params.get('path', '')

        self.port = self._make_port(params)
        self.query = self._make_query(endpoint, priority_params)
        self.netloc = self._make_netloc()

    @staticmethod
    def _dict_to_tuples(d):
        try:
            iter_ = d.lists
        except AttributeError:
            iter_ = d.items

        return ((k, v if isinstance(v, (list,)) else [v]) for k, v in iter_())

    def _make_query(self, endpoint, params):
        query = utils.QueryDict()
        query.update(MultiValueDict(self._dict_to_tuples(endpoint.get('query') or {})))
        query.update(MultiValueDict(self._dict_to_tuples(params.get('query') or {})))
        return query.urlencode(safe=params.get('safe')) if query else None

    def _make_port(self, params):
        port = params.get('port', '')
        if not port:
            return None

        port = int(port)

        if self.schema == 'https' and port == 443:
            return None

        if self.schema == 'http' and port == 80:
            return None

        return port

    def _make_netloc(self):
        if self.port is None:
            return self.host
        else:
            return '{self.host}:{self.port}'.format(self=self)

    def base_url(self):
        return urlunsplit((self.schema, self.netloc, '', '', ''))

    def url(self):
        return urlunsplit((self.schema, self.netloc, self.path, self.query, ''))


class Endpoint(dict):
    def base_url(self):
        builder = UrlBuilder(self)
        return builder.base_url()

    def url(self, **kwargs):
        builder = UrlBuilder(self, kwargs)
        return builder.url()

    def headers(self, **kwargs):
        headers = copy.copy(self.get('headers', {}))

        token_type = self.get('oauth_token')
        if token_type is not None:
            token = get_oauth_token(token_type)
            headers.update(get_oauth_header(token))

        token_type = self.get('x-api-key')
        if token_type is not None:
            token = get_oauth_token(token_type)
            headers.update(get_x_api_key_header(token))

        headers.update(kwargs)

        return headers
