import pytz
from pydantic import BaseSettings, SecretStr
from tvmauth import BlackboxTvmId as BlackboxClientId
from celery.schedules import crontab


class Settings(BaseSettings):
    ENV_TYPE: str = 'default'
    DEBUG: bool = True

    DEFAULT_TIMEZONE = pytz.timezone('Europe/Moscow')

    CELERY_TIMEZONE = pytz.timezone('Europe/Moscow')

    CA_ROOT_PATH: str = '/etc/ssl/certs/ca-certificates.crt'

    POSTGRES_DATABASE: str = 'stackbot'
    POSTGRES_USER: str = 'postgres'
    POSTGRES_PASSWORD: str = 'postgres'
    POSTGRES_HOST: str = 'localhost'
    POSTGRES_PORT: int = 5432

    @property
    def db_settings(self):
        return {
            'database': self.POSTGRES_DATABASE,
            'user': self.POSTGRES_USER,
            'password': self.POSTGRES_PASSWORD,
            'host': self.POSTGRES_HOST,
            'port': self.POSTGRES_PORT,
        }

    @property
    def database_url(self):
        return 'postgresql://{user}:{password}@{host}:{port}/{database}'.format(
            **self.db_settings
        )

    CELERY_BROKER_TRANSPORT_OPTIONS: dict = {
        'is_secure': False,
        'region': 'yandex',
        'visibility_timeout': 60 * 60
    }
    CELERY_BROKER_URL: str = 'sqs://sqs.yandex.net:8771'
    CELERY_WORKER_HIJACK_ROOT_LOGGER: bool = False
    CELERY_BEAT_SCHEDULE = {
        'synchronize_bot_users': {
            'task': 'stackbot.tasks.synchronize_bot_users',
            'schedule': crontab(minute=0, hour=12),
        },
        'process_recent_active_questions': {
            'task': 'stackbot.tasks.process_recent_active_questions',
            'schedule': crontab(),
        },
        'send_question_posted_event': {
            'task': 'stackbot.tasks.send_question_posted_event',
            'schedule': crontab(),
        },
        'send_unanswered_digest': {
            'task': 'stackbot.tasks.send_unanswered_digest',
            'schedule': crontab(hour=11, day_of_week=5),
        },
    }

    EMAIL_HOST: str = 'outbound-relay.yandex.net'
    EMAIL_PORT: int = 25
    EMAIL_ADDR: str = 'robot-stackbot@yandex-team.ru'

    STACK_API_HOSTS: list = [
        'https://so-app-b.stackoverflow.yandex-team.ru/',
        'https://so-app-a.stackoverflow.yandex-team.ru/',
    ]
    STAFF_HOST: str = 'https://staff.yandex-team.ru'
    STAFF_API_HOST: str = 'https://staff-api.yandex-team.ru'
    NO_AUTH_MESSAGE: str = 'Действие запрещено, так как отсутствует привязка учетной записи'

    TELEGRAM_TOKEN: str
    STAFF_API_TOKEN: str
    STACK_API_KEY: str
    STACK_ACCESS_TOKEN: str

    YT_TOKEN: str = 'fake_token'

    TVM2_CLIENT: str
    TVM2_SECRET: SecretStr
    TVM2_BLACKBOX_CLIENT = BlackboxClientId.ProdYateam
    TVM2_SERVICES: dict = {
        'gozora': 2023123,
        'startrek': 177,
    }

    LOGBROKER_TOKEN: str
    LOGBROKER_TOPIC: str
    LOGBROKER_PRODUCER: str
    LOGBROKER_PORT: int = 2135
    LOGBROKER_ENDPOINT: str = 'logbroker.yandex.net'
    LOGBROKER_CLIENT_WAIT_TIMEOUT: int = 10
    LOGBROKER_CLIENT_MAX_INFLIGHT: int = 50

    USE_GOZORA: bool = False

    TASK_QUESTIONS_SYNC_GAP_IN_MINS: int = 5

    # Фильтр questions: берет question_id, tags, last_activity_date, creation_date
    STACK_API_FILTER_QUESTION_TAGS: str = '!)riR7Ef(W0I0-fo0bvKb'
    # Фильтр questions/{ids}: берет question_id, title, body, body_markdown, last_activity_date, tags
    STACK_API_FILTER_QUESTION_BODY: str = '!)riR7Z9)aTWl0OP8kO7f'
    # Фильтр /events : берем event_id, event_type, creation_date, link
    STACK_API_FILTER_EVENTS: str = '!nKzQUR8uLc'
    # Фильтр /answers : берем answer_id, is_accepted, owner.user_id, owner.display_name
    STACK_API_FILTER_ANSWER_ACCEPTED: str = '!SXs3rpp*rDRdeFzfvE'
    # Фильтр /questions : берем вопросы с ответами и комменатриями к ним и доп информацию по лайкам/дизлайкам
    STACK_API_FILTER_QUESTIONS_ALL_DATA: str = '!7uzPYKI9qtB_M4CRz_JNS4.B8DKw2mPqOy'
    # Фильрт /articles : берем статьи с комментариями к ним
    STACK_API_FILTER_ARTICLES_ALL_DATA: str = '!*MZqiH3*6SK8BEW6'

    STAFF_ACHIEVEMENTS_X_ACCEPTED_ANSWERS: list[tuple[int, int]] = [
        (5, 4624), (40, 4623), (100, 4626),
    ]

    STACKOVERFLOW_HOST: str = 'https://stackoverflow.yandex-team.ru'
    STARTREK_API_HOST: str = 'https://st-api.yandex-team.ru'
    STARTREK_QUEUE: str = 'STACKTEST'
    STARTREK_API_TOKEN: str
    STARTREK_PEPPER: str = 'pepper'

    ISSUE_TYPE_QUESTION: int = 66
    COMPONENT_ID_QUESTION: int = 112761
    COMPONENT_ID_ANSWER: int = 112762
    SO_UPVOTES_ID: str = '625d6a7611d3af3e89c1eda3--soUpvotes'

    ROBOT_LOGIN = 'robot-stackbot'

    class Config:
        env_prefix = ''
