from sqlalchemy import (
    Column,
    ForeignKey,
    types,
)
from sqlalchemy.orm import (
    relationship,
    backref,
)

from stackbot.enums import (
    SubscriptionEventState,
    ChatType,
    SubscriptionType,
)
from stackbot.db.base import (
    BaseModel,
    TimestampedModelMixin,
)


class Chat(TimestampedModelMixin, BaseModel):
    __tablename__ = 'chat'

    id = Column(types.Integer, primary_key=True, autoincrement=False)  # chat id
    title = Column(types.String, nullable=False)
    chat_type = Column(
        types.Enum(ChatType),
        nullable=False,
    )

    author_id = Column(types.Integer, ForeignKey('bot_user.id'), nullable=False, )
    author = relationship('BotUser', foreign_keys=[author_id])


class Email(TimestampedModelMixin, BaseModel):
    __tablename__ = 'email'

    id = Column(types.Integer, primary_key=True, index=True)
    address = Column(types.String, nullable=False)

    author_id = Column(types.Integer, ForeignKey('bot_user.id'), nullable=False, )
    author = relationship('BotUser', foreign_keys=[author_id])


class Subscription(TimestampedModelMixin, BaseModel):
    __tablename__ = 'subscription'

    id = Column(types.Integer, primary_key=True, index=True)
    tag = Column(types.String, nullable=False, index=True)
    type = Column(types.Enum(SubscriptionType), nullable=False)

    chat_id = Column(types.Integer, ForeignKey('chat.id', ondelete='CASCADE'), nullable=True, )
    chat = relationship('Chat', foreign_keys=[chat_id], backref=backref('subscriptions', cascade='all,delete'))

    email_id = Column(types.Integer, ForeignKey('email.id', ondelete='CASCADE'), nullable=True, )
    email = relationship('Email', foreign_keys=[email_id], backref=backref('subscriptions', cascade='all,delete'))


class SubscriptionUnanswered(TimestampedModelMixin, BaseModel):
    __tablename__ = 'subscription_unanswered'

    id = Column(types.Integer, primary_key=True, index=True)
    tag = Column(types.String, nullable=False, index=True)
    type = Column(types.Enum(SubscriptionType), nullable=False)

    chat_id = Column(types.Integer, ForeignKey('chat.id', ondelete='CASCADE'), nullable=True, )
    chat = relationship('Chat', foreign_keys=[chat_id], backref=backref('subscriptions_digest', cascade='all,delete'))

    email_id = Column(types.Integer, ForeignKey('email.id', ondelete='CASCADE'), nullable=True, )
    email = relationship('Email', foreign_keys=[email_id], backref=backref('subscriptions_digest', cascade='all,delete'))


class SubscriptionToEvent(TimestampedModelMixin, BaseModel):
    __tablename__ = 'subscription_to_event'

    id = Column(types.Integer, primary_key=True, index=True)

    subscription_id = Column(types.Integer, ForeignKey('subscription.id', ondelete='CASCADE'), nullable=False,)
    subscription = relationship('Subscription', foreign_keys=[subscription_id])

    event_id = Column(types.Integer, ForeignKey('event.id'), nullable=False, )
    event = relationship('Event', foreign_keys=[event_id])

    state = Column(
        types.Enum(SubscriptionEventState),
        nullable=False,
        default=SubscriptionEventState.new,
    )
