import logging
from collections import defaultdict

from stackbot.config import settings
from stackbot.logic.utils import get_chunks
from .base import BaseClient

logger = logging.getLogger(__name__)


class AchieveryClient(BaseClient):
    HOST = settings.STAFF_HOST
    OAUTH_TOKEN = settings.STAFF_API_TOKEN

    def _get_current_achievements(self, staff: list[str]) -> list:
        result = []
        achievement_ids = [_id for count, _id in settings.STAFF_ACHIEVEMENTS_X_ACCEPTED_ANSWERS]
        for chunk in get_chunks(staff, size=10):
            page = 1
            has_more = True
            while has_more:
                response = self._make_request(
                    path='/api/achievery/given/',
                    params={
                        '_limit': 100,
                        '_page': page,
                        'person.login': chunk,
                        'achievement.id': achievement_ids,
                        '_fields': ','.join(['id', 'is_active', 'revision', 'person.login', 'achievement.id']),
                    },
                    method='get'
                )
                response_data = response.json()
                has_more = page != response_data['pages']
                page += 1
                result.extend(response_data.get('objects', []))
        return result

    def _update_given_achievement(self, data: list[dict]) -> None:
        for item in data:
            self._make_request(
                path='/api/achievery/given/{}/'.format(item['id']),
                data={
                    'is_active': item['is_active'],
                    'revision': item['revision'],
                },
                method='put'
            )

    def _give_achievement(self, data: list[dict]) -> None:
        for item in data:
            self._make_request(
                path='/api/achievery/given/',
                data={
                    'level': '-1',
                    'achievement.id': item['achievement_id'],
                    'person.login': item['staff'],
                },
                method='post'
            )

    def update_staff_achievements(self, staff_achievements: dict[str, int]) -> None:
        current_achievements = defaultdict(list)

        result = self._get_current_achievements(staff=list(staff_achievements.keys()))
        for achievement in result:
            current_achievements[achievement['person']['login']].append(achievement)

        to_update = list()
        to_create = list()
        for staff in current_achievements:
            found_necessary_achievement = False
            for given in current_achievements[staff]:
                # соответствующая уровню ачивка не активирована - активируем
                if given['achievement']['id'] == staff_achievements[staff]:
                    found_necessary_achievement = True
                    if not given['is_active']:
                        to_update.append({
                            'is_active': True, 'id': given['id'], 'revision': given['revision'],
                        })
                # ачивка не соответствует уровню - деактивируем
                else:
                    if given['is_active']:
                        to_update.append({
                            'is_active': False, 'id': given['id'], 'revision': given['revision'],
                        })
            # у стаффа нет нужной ачивки - добавляем
            if not found_necessary_achievement:
                to_create.append({
                    'achievement_id': staff_achievements[staff], 'staff': staff,
                })

        self._give_achievement(to_create)
        self._update_given_achievement(to_update)


achievery_client = AchieveryClient()
