import urllib.parse
import logging

from typing import Optional
from requests import Response, HTTPError
from requests.sessions import Session as BaseSession
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from stackbot.config import settings

logger = logging.getLogger(__name__)

BACKOFF_FACTOR = 0.2
STATUS_FORCELIST = [429, 499, 500, 502, 503, 504]
CONNECT_RETRIES = 3
READ_RETRIES = 3


class BaseClient:
    TVM2_CLIENT = None
    OAUTH_TOKEN = None
    X_UID_HEADER = False
    HOST = None

    def __init__(
        self,
        **kwargs
    ):
        self.host = kwargs.get('host') or self.HOST
        self.kwargs = kwargs

    def get_headers(self, headers: Optional[dict] = None) -> dict:
        result_headers = {}
        if headers:
            result_headers.update(headers)
        if self.OAUTH_TOKEN:
            result_headers['Authorization'] = f'OAuth {self.OAUTH_TOKEN}'
        # if self.TVM2_CLIENT:
        #    result_headers['X-Ya-Service-Ticket'] = get_tvm2_ticket(self.TVM2_CLIENT)
        if self.X_UID_HEADER:
            result_headers['X-UID'] = settings.ROBOT_UID
        return result_headers

    def _raise_for_status(self, response: Response) -> None:
        try:
            response.raise_for_status()
        except HTTPError as exc:
            logger.error(f"{exc.args[0]}: {exc.response.text}")
            raise

    def _make_request(
        self, path: str,
        params: Optional[dict] = None, headers: Optional[dict] = None,
        data: Optional[dict] = None, raw_data: Optional[bytes] = None,
        method: str = 'get', status_codes_exclude_from_raise: Optional[set] = None,
    ) -> Response:
        url = urllib.parse.urljoin(self.host, path)
        logger.info(f'Making request in {self.__class__.__name__}')
        with Session(headers=self.get_headers(headers)) as session:
            response = getattr(session, method)(url, params=params, json=data, data=raw_data)
        logger.info(f'Got response from {self.__class__.__name__}, status_code: {response.status_code}')

        if status_codes_exclude_from_raise and response.status_code not in status_codes_exclude_from_raise:
            self._raise_for_status(response=response)

        return response


class Session(BaseSession):
    def __init__(
        self,
        verify: Optional[bool] = None,
        headers: Optional[dict] = None,
        connect_retries=CONNECT_RETRIES,
        read_retries=READ_RETRIES,
    ):
        super(Session, self).__init__()

        if verify is None:
            verify = settings.CA_ROOT_PATH

        self.verify = verify

        if headers is not None:
            self.headers.update(headers)

        retry = Retry(
            connect=connect_retries,
            read=read_retries,
            backoff_factor=BACKOFF_FACTOR,
            redirect=0,
            raise_on_status=True,
            raise_on_redirect=True,
            status_forcelist=STATUS_FORCELIST,
        )

        adapter = HTTPAdapter(max_retries=retry)
        self.mount('http://', adapter)
        self.mount('https://', adapter)
