import logging
import urllib.parse
from collections import defaultdict

from datetime import datetime, timedelta
from requests.exceptions import RequestException
from typing import Optional, Tuple

from .base import BaseClient
from stackbot.config import settings
from stackbot.logic.utils import get_chunks

logger = logging.getLogger(__name__)


class StackClient(BaseClient):
    HOSTS = settings.STACK_API_HOSTS
    API_VERSION = '2.3'

    def get_headers(self, headers: Optional[dict] = None) -> dict:
        headers = super(StackClient, self).get_headers(headers=headers)
        headers['X-API-Key'] = settings.STACK_API_KEY
        headers['X-API-Access-Token'] = settings.STACK_ACCESS_TOKEN
        headers['Host'] = 'stackoverflow.yandex-team.ru'
        return headers

    def _make_request(self, *args, **kwargs):
        for host in self.HOSTS:
            try:
                self.host = host
                return super(StackClient, self)._make_request(*args, **kwargs)
            except RequestException as exc:
                last_exc = exc
        raise last_exc

    def _get_tag(self, tag_str: str) -> dict:
        return self._make_request(
            path=f'api/{self.API_VERSION}/tags/{tag_str}/info', method='get',
        ).json()

    def check_if_tag_exists(self, tag: str) -> Tuple[bool, Optional[list[dict]]]:
        response = self._get_tag(tag)
        if any({tag_info['name'] == tag for tag_info in response['items']}):
            return True, None
        return False, response['items']

    def get_existing_tags(self, tags: set[str]) -> set[str]:
        response = self._get_tag(';'.join(tags))
        if 'items' not in response:
            return set()
        existing_tags = {tag['name'] for tag in response['items']}
        return tags.intersection(existing_tags)

    def get_questions(self, ids: list) -> list:
        result = []
        for chunk in get_chunks(ids, size=100):
            ids = ';'.join(map(str, chunk))
            response = self._make_request(
                path=f'api/{self.API_VERSION}/events/{ids}',
                params={'pagesize': 100},
                method='get',
            )
            response_data = response.json()
            result.extend(response_data['items'])
        return result

    def get_questions_info(self, ids: list) -> dict:
        result = {}
        for chunk in get_chunks(ids, size=100):
            ids = ';'.join(map(str, chunk))
            response = self._make_request(
                path=f'api/{self.API_VERSION}/questions/{ids}',
                params={
                    'pagesize': 100,
                    'filter': settings.STACK_API_FILTER_QUESTION_BODY
                },
                method='get',
            )
            response_data = response.json()
            result.update({
                question['question_id']: question
                for question in response_data['items']
            })
        return result

    def unanswered_questions_by_tags(self, tags: set[str]) -> dict:
        result = {}
        for tag in tags:
            has_more = True
            page = 1
            while has_more:
                response = self._make_request(
                    path=f'api/{self.API_VERSION}/questions',
                    params={
                        'pagesize': 100,
                        'page': page,
                        'tagged': tag,
                        'filter': '!Dhka-sUqJ-EWR5NnY',  # https://jing.yandex-team.ru/files/smosker/2022-02-17_13-57-47.png
                    },
                    method='get',
                )
                response_data = response.json()
                page += 1
                has_more = response_data.get('has_more')

                for item in response_data['items']:
                    if item['answer_count'] > 0:
                        continue
                    result[item['link']] = item
        return result

    def get_recent_active_questions(self, gap_in_mins: int) -> list:
        utc_start = int((datetime.utcnow() - timedelta(minutes=gap_in_mins)).timestamp())
        result = []
        has_more = True
        page = 1
        while has_more:
            response = self._make_request(
                path=f'api/{self.API_VERSION}/questions',
                params={
                    'order': 'desc',
                    'sort': 'activity',
                    'pagesize': 50,
                    'page': page,
                    'filter': settings.STACK_API_FILTER_QUESTION_TAGS,
                },
                method='get',
            )
            response_data = response.json()
            page += 1
            has_more = response_data.get('has_more')

            for item in response_data['items']:
                if item['last_activity_date'] < utc_start:
                    has_more = False
                    break
                result.append(item)

        return result

    def _all_items(self, item_name: str, _filter: Optional[str] = 'default') -> list:
        result = []
        has_more = True
        page = 1
        while has_more:
            response = self._make_request(
                path=f'api/{self.API_VERSION}/{item_name}/',
                params={
                    'page': page,
                    'pagesize': 100,
                    'filter': _filter,
                },
                method='get',
            )
            response_data = response.json()
            result.extend(response_data['items'])
            page += 1
            has_more = response_data.get('has_more')
        return result

    def all_posts(self) -> tuple[list, list]:
        articles = self._all_items('articles', _filter=settings.STACK_API_FILTER_ARTICLES_ALL_DATA)
        questions_with_answers = self._all_items('questions', _filter=settings.STACK_API_FILTER_QUESTIONS_ALL_DATA)
        return articles, questions_with_answers

    def all_answers(self) -> list:
        return self._all_items('answers')

    def all_questions(self) -> list:
        return self._all_items('questions')

    def all_users(self) -> list:
        return self._all_items('users')

    def get_events(self) -> list:
        return self._all_items('events', _filter=settings.STACK_API_FILTER_EVENTS)

    def get_user_by_login(self, login: str) -> Optional[dict]:
        response = self._make_request(
            path=f'api/{self.API_VERSION}/users/',
            params={
                'inname': login,
            },
            method='get',
        )
        response_data = response.json()
        for user in response_data['items']:
            if user.get('display_name') == login:
                return user

    def search_questions(self, param: str, limit: int) -> list[dict]:
        response = self._make_request(
            path=f'api/{self.API_VERSION}/search/advanced/',
            params={
                'q': param,
                'sort': 'relevance',
                'order': 'desc',
                'pagesize': min(limit, 100),
            },
            method='get',
        )
        response_data = response.json()
        return response_data['items']

    def create_question(self, title: str, tags: list[str], body: str) -> Tuple[Optional[str], Optional[str]]:
        params = {
            'title': title,
            'tags': ';'.join(tags),
            'body': body,
        }
        encoded_params = '&'.join('{}={}'.format(
            k, urllib.parse.quote(str(v).encode('utf-8'), safe='')
        ) for k, v in params.items()).encode('utf-8')

        response = self._make_request(
            path=f'api/{self.API_VERSION}/questions/add',
            raw_data=encoded_params,
            method='post',
            headers={
                'Content-Type': 'application/x-www-form-urlencoded',
                'X-Forwarded-For': '127.0.0.1',
            },
            status_codes_exclude_from_raise={400},  # stackoverflow кидает 400 если вопрос не проходит проверку
        )
        response_data = response.json()
        if response.status_code == 200:
            return response_data['items'][0]['link'], None
        elif response.status_code == 400:
            logger.warning(f'Got {response.content} response from stack')
            return None, response_data['error_message']

    def get_users_accepted_answer_counts(self) -> defaultdict[str, int]:
        accepted_counts = defaultdict(int)
        result = []
        has_more = True
        page = 1
        while has_more:
            response = self._make_request(
                path=f'api/{self.API_VERSION}/answers',
                params={
                    'filter': settings.STACK_API_FILTER_ANSWER_ACCEPTED,
                    'pagesize': 100,
                    'page': page,
                },
                method='get',
            )
            response_data = response.json()
            result.extend(response_data['items'])
            page += 1
            has_more = response_data.get('has_more')

        for item in result:
            if item['owner']['user_type'] not in ('registered', 'moderator', 'team_admin'):
                continue
            if not item['is_accepted']:
                continue

            login = item['owner']['display_name']
            accepted_counts[login] += 1

        return accepted_counts
