import logging
from collections import defaultdict
from datetime import datetime

import startrek_client
from startrek_client.settings import VERSION_V2

from stackbot.config import settings


logger = logging.getLogger(__name__)


class Queues(startrek_client.collections.Queues):
    _priority = 0.6

    @startrek_client.collections.injected_property
    def tags(self, queue):
        return self._execute_request(
            self._connection.get,
            path=queue._path + '/tags',
        )


class Startrek(startrek_client.Startrek):
    def __init__(self, *args, **kwargs):
        super(Startrek, self).__init__(*args, **kwargs)
        self.queues = self._get_collection(Queues)


class StartrekClient:
    def __init__(self):
        self.startrek = Startrek(
            api_version=VERSION_V2,
            useragent='stackbot',
            base_url=settings.STARTREK_API_HOST,
            token=settings.STARTREK_API_TOKEN,
        )
        self.queue = settings.STARTREK_QUEUE
        self._timestamp_cached_tags, self._cached_tags = None, None

    def _get_tags(self):
        timestamp = datetime.now().timestamp()
        if not self._cached_tags or timestamp - self._timestamp_cached_tags >= 180:
            self._timestamp_cached_tags = timestamp
            self._cached_tags = self.startrek.queues[self.queue].tags
        return self._cached_tags

    def check_if_tag_exists(self, tag: str) -> bool:
        return tag in set(self._get_tags())

    def get_existing_tags(self, tags: set[str]) -> set[str]:
        return tags.intersection(set(self._get_tags()))

    def get_questions_info(self, keys: list[int]) -> dict:
        real_keys = [f'{self.queue}-{key}' for key in keys]
        issues = self.startrek.issues.find(
            filter={
                'queue': self.queue,
                'components': settings.COMPONENT_ID_QUESTION,
                'key': real_keys,
            },
            expand=['links'],
            order=['-updated'],
            per_page=100,
        )
        result = {}
        for issue in issues:
            result[int(issue.key.partition('-')[2])] = issue
        return result

    def unanswered_questions_by_tags(self, tags: set[str]) -> dict:
        result = {}
        # Здесь получаем не вопросы без принятых ответов,
        # а вопросы без каких-либо ответов в принципе
        issues = self.startrek.issues.find(
            query=' '.join([
                f'Queue: {self.queue} AND',
                f'Components: {settings.COMPONENT_ID_QUESTION} AND (',
                'OR'.join(f'Tags: "{tag}"' for tag in tags),
                ') "Relates": empty() AND "Depends On": empty()',
                '"Sort By": Updated DESC',
            ]),
            expand=['links'],
            per_page=100,
        )
        for issue in issues:
            pos = issue.description.rfind('@')
            # И статьи и вопросы имеют одинаковый компонент "Вопрос",
            # различить их можно только по доп.информации в конце описания тикета:
            if pos != -1 and issue.description[pos-8:pos-1] == 'вопроса':
                result[issue.key] = issue
        return result

    def get_recent_active_questions(self, gap_mins: int) -> list:
        return self.startrek.issues.find(
            query=' '.join([
                f'Queue: {self.queue} AND',
                f'Components: {settings.COMPONENT_ID_QUESTION} AND',
                f'Updated: >=now()-"{gap_mins}m"',
            ]),
            per_page=100,
        )

    def search_questions(self, param: str, limit: int) -> list:
        issues = self.startrek.issues.find(
            query='Queue: {} AND Components: {} AND (Summary: "{}" OR Description: "{}" OR Tags: {})'.format(
                self.queue, settings.COMPONENT_ID_QUESTION,
                param, param, ','.join(f'"{tag}"' for tag in param.split(' '))
            ),
            per_page=min(limit, 100),
        )
        result = []
        for issue in issues:
            result.append(issue)
            if len(result) >= limit:
                break
        return result

    def create_question(self, title: str, tags: list[str], body: str):
        return self.startrek.issues.create(
            queue=settings.STARTREK_QUEUE,
            summary=title,
            type={'id': settings.ISSUE_TYPE_QUESTION},
            description=body,
            tags=list(tags),
            components=settings.COMPONENT_ID_QUESTION,
        )

    def get_users_accepted_answer_counts(self) -> defaultdict[str, int]:
        accepted_counts = defaultdict(int)
        issues = self.startrek.issues.find(
            filter={
                'queue': self.queue,
                'components': settings.COMPONENT_ID_ANSWER,
            },
            expand=['links'],
            order=['-updated'],
            per_page=100,
        )

        for issue in issues:
            if not len(list(issue.links)):
                logger.warning('Answer has no linked question: %s', issue.key)
                continue
            res = issue.description.rsplit('@', maxsplit=1)
            if not len(res) == 2:
                logger.warning('Answer has no author specified: %s', issue.key)
                continue
            _, login = res
            if not login.replace('-', '').isalnum():
                logger.warning('Issue %s author\'s Login has illicit characters: %s', issue.key, login)
                continue

            link = issue.links[0]
            if link.type.id == 'depends':
                accepted_counts[login] += 1

        return accepted_counts

    @staticmethod
    def get_link(issue) -> str:
        return f'{settings.STACKOVERFLOW_HOST}/questions/{issue.key}'


startrek_client = StartrekClient()
