import logging
from typing import Optional

from sqlalchemy.orm import Session, Query, joinedload
from telegram.chat import Chat as TelegramChat

from stackbot.enums import (
    BotUserState,
    ChatType,
    SubscriptionType,
)
from stackbot.db import (
    BaseModel,
    BotUser,
    Subscription,
    Chat,
    Email,
)


logger = logging.getLogger(__name__)


def get_bot_user_by_id(db: Session, telegram_id: int) -> Optional[BotUser]:
    return (
        db.query(BotUser)
        .filter(BotUser.telegram_id == str(telegram_id))
        .first()
    )


def create_bot_user(
    db: Session, staff_info: 'StaffInfo', telegram_id: int, username: str
) -> BotUser:
    bot_user = BotUser(
        telegram_id=telegram_id,
        username=username,
        state=BotUserState.active,

        staff_id=staff_info.id,
        staff_uid=staff_info.uid,
        staff_login=staff_info.login,
    )
    db.add(bot_user)
    db.commit()
    db.refresh(bot_user)
    return bot_user


def get_or_create_bot_user(
    db: Session, staff_info: 'StaffInfo', telegram_id: int, username: str
) -> BotUser:

    bot_user = get_bot_user_by_id(db, telegram_id=telegram_id)
    if bot_user is None:
        bot_user = create_bot_user(
            db, staff_info=staff_info, telegram_id=telegram_id, username=username
        )
    else:
        if bot_user.state == BotUserState.deprived:
            bot_user.state = BotUserState.active
            db.add(bot_user)
            db.commit()
            db.refresh(bot_user)
    return bot_user


def get_subscription(db: Session, tag: str, chat_id: int, model: BaseModel = Subscription) -> Optional[Subscription]:
    return db.query(model).filter(
        model.tag == tag,
        model.chat_id == chat_id
    ).first()


def get_email_subscription(db: Session, tag: str, email_address: str, model: BaseModel = Subscription) -> Optional[Subscription]:
    return (
        db.query(model)
        .join(Email, Email.id == model.email_id)
        .filter(
            model.tag == tag,
            Email.address == email_address
        )
        .options(joinedload(model.email).joinedload(Email.author))
        .first()
    )


def query_subscriptions_by_chat(db: Session, chat_id: int, model: BaseModel = Subscription) -> Query:
    return db.query(model).filter(
        model.chat_id == chat_id
    )


def query_email_subscriptions(db: Session, author_id: int, model: BaseModel = Subscription) -> Query:
    return (
        db.query(model)
        .join(Email, Email.id == model.email_id)
        .filter(Email.author_id == author_id)
    )


def delete_subscriptions(db: Session, subscription_ids: list[int], model: BaseModel = Subscription) -> None:
    db.query(model).filter(model.id.in_(subscription_ids)).delete(synchronize_session=False)


def create_subscription(db: Session, tag: str, chat: Chat, model: BaseModel = Subscription) -> BaseModel:
    subscription = model(
        chat_id=chat.id,
        tag=tag,
        type=SubscriptionType.chat,
    )
    db.add(subscription)
    db.commit()
    db.refresh(subscription)
    return subscription


def create_email_subscription(db: Session, tag: str, email: Email, model: BaseModel) -> Subscription:
    subscription = model(
        email_id=email.id,
        tag=tag,
        type=SubscriptionType.email,
    )
    db.add(subscription)
    db.commit()
    db.refresh(subscription)
    return subscription


def get_or_create_chat(db: Session, chat_data: TelegramChat, author_id: int) -> Chat:
    chat = db.query(Chat).get(chat_data.id)
    if not chat:
        title = get_chat_name(chat=chat_data)
        chat = Chat(
            id=chat_data.id,
            author_id=author_id,
            title=title,
            chat_type=chat_data.type,
        )
        db.add(chat)
        db.commit()
        db.refresh(chat)
    return chat


def get_or_create_email(db: Session, email_address: str, author_id: int) -> Email:
    email = db.query(Email).filter(Email.address == email_address, Email.author_id == author_id).first()
    if not email:
        email = Email(
            author_id=author_id,
            address=email_address,
        )
        db.add(email)
        db.commit()
        db.refresh(email)
    return email


def get_chat_name(chat: TelegramChat) -> str:
    if chat.type in ChatType.group_types():
        title = chat.title
    else:
        title = chat.username
    return title


def get_chunks(iterable, size=20):
    """
    Разбивает iterable на группы по size элементов в каждом.
    Если длина iterable не кратна size,
    то длина последнего куска будет меньше size.
    >>> list(get_chunks(range(8), 4))
    [[0, 1, 2, 3], [4, 5, 6, 7]]
    >>> list(get_chunks(range(8), 3))
    [[0, 1, 2], [3, 4, 5], [6, 7]]
    """
    for i in range(0, len(iterable), size):
        yield iterable[i:i + size]
