import datetime
from collections import defaultdict
import functools
import inspect
from typing import Optional, Callable
import argparse
import logging
from logging.config import dictConfig
import html
import re
import urllib.parse

from telegram import Update, Bot, ParseMode
from telegram.utils.request import Request
from telegram.ext import (
    Updater,
    Filters,
    CallbackContext,
    CommandHandler,
    MessageHandler,
    ConversationHandler,
)
from sqlalchemy.orm import Session

from stackbot.config import settings
from stackbot.db import (
    dbconnect, BotUser,
    Subscription, SubscriptionUnanswered,
)
from stackbot.enums import (
    BotUserState,
    ChatType,
    AskStates,
)
from stackbot.so_migrate_to_st import find_imported_subs, take_ownership
from stackbot.logconfig import LOGGING_CONFIG
from stackbot.logic import utils
from stackbot.logic.tvm_client import get_tvm2_ticket
from stackbot.logic.clients.staff import staff_client
from stackbot.logic.clients.startrek import startrek_client


dictConfig(LOGGING_CONFIG)
logger = logging.getLogger(__name__)


class StackRequest(Request):
    def _request_wrapper(self, method, url, *args, **kwargs):
        headers = kwargs.get('headers') or {}
        headers['x-ya-service-ticket'] = get_tvm2_ticket('gozora')
        headers['x-ya-client-id'] = 'stack_bot'
        headers['x-ya-dest-url'] = url

        kwargs['headers'] = headers

        url = 'http://go.zora.yandex.net:1080'

        return super()._request_wrapper(method, url, *args, **kwargs)


class StackBot(Bot):
    pass


def get_bot(token: str) -> Bot:
    if settings.USE_GOZORA:
        request = StackRequest(con_pool_size=5)
        bot = StackBot(token, request=request)
    else:
        request = Request(con_pool_size=5)
        bot = Bot(token, request=request)
    return bot


def auth_handler(func: Callable):
    def filter_args(args, fn):
        params = [
            param.name for param in inspect.signature(fn).parameters.values()
            if param.kind == param.POSITIONAL_OR_KEYWORD
        ]
        return {name: args[name] for name in params}

    @functools.wraps(func)
    @dbconnect
    def wrapper(self, update: Update, context: CallbackContext, **kwargs):
        state, message = None, None
        try:
            session = kwargs.get('session')
            bot_user = utils.get_bot_user_by_id(db=session, telegram_id=update.effective_user.id)
            message = None
            if bot_user is None or bot_user.state != BotUserState.active:
                message = 'Not authorized, use: `/start`'
            else:
                kwargs.update({
                    'self': self, 'bot_user': bot_user, 'context': context, 'update': update,
                })
                kwargs = filter_args(kwargs, func)
                result = func(**kwargs)
                if type(result) == tuple:
                    state, message = result[:2]
                else:
                    message = result

        except Exception:
            message = "Unexpected error occurred"
            raise
        finally:
            context.bot.send_message(
                chat_id=update.effective_chat.id, text=message, parse_mode=ParseMode.MARKDOWN,
            )
        return state

    return wrapper


def unescape_body(text: str) -> str:
    result = []
    prev_r = 0
    for match in re.finditer(r'(```|`).*?\1', string=text, flags=re.DOTALL):  # captures ```code blocks``` and `inline code`
        l, r = match.start(), match.end()
        result.append(text[prev_r:l])
        result.append(re.sub(r'\\([\*\[\_`])', r'\1', text[l:r]))  # \\* -> *, \\` -> `, \\_ -> _, \\[ -> [
        prev_r = r
    result.append(text[prev_r:])
    return ''.join(result)


class BotHandler:
    def __init__(self, token: Optional[str] = None):
        if not token:
            token = settings.TELEGRAM_TOKEN
        kwargs = {
            'workers': 1,
        }
        if settings.USE_GOZORA:
            kwargs['bot'] = get_bot(token)
        else:
            kwargs['token'] = token

        self.updater = Updater(**kwargs)
        self._add_handlers()

    def _add_handlers(self):
        dispatcher = self.updater.dispatcher

        dispatcher.add_handler(CommandHandler(command='start', callback=self._start))
        dispatcher.add_handler(CommandHandler(command='list_subscriptions', callback=self._list_subscriptions))
        dispatcher.add_handler(CommandHandler(command='subscribe', callback=self._subscribe))
        dispatcher.add_handler(CommandHandler(command='unsubscribe', callback=self._unsubscribe))
        dispatcher.add_handler(CommandHandler(command='unsubscribe_all', callback=self._unsubscribe_all))
        dispatcher.add_handler(CommandHandler(command='subscribe_email', callback=self._subscribe_email))
        dispatcher.add_handler(CommandHandler(command='unsubscribe_email', callback=self._unsubscribe_email))
        dispatcher.add_handler(CommandHandler(command='search', callback=self._search))

        dispatcher.add_handler(ConversationHandler(
            entry_points=[
                CommandHandler(command='ask', callback=self._ask_start),
            ],
            states={
                AskStates.REPLY_TITLE: [
                    MessageHandler(filters=(Filters.text & ~Filters.command), callback=self._ask_reply_title),
                ],
                AskStates.REPLY_TAGS: [
                    MessageHandler(filters=(Filters.text & ~Filters.command), callback=self._ask_reply_tags),
                ],
                AskStates.REPLY_BODY: [
                    MessageHandler(
                        filters=((Filters.text | Filters.forwarded | Filters.reply) & ~Filters.command),
                        callback=self._ask_reply_body
                    )
                ],
                AskStates.PREVIEW_QUESTION: [
                    CommandHandler(command='done', callback=self._ask_finish),
                ],
                AskStates.CONFIRM_TAGS: [
                    CommandHandler(command='confirm', callback=self._ask_confirm_tags),
                    MessageHandler(filters=(Filters.text & ~Filters.command), callback=self._ask_reply_tags),
                ],
                ConversationHandler.TIMEOUT: [
                    MessageHandler(filters=Filters.text, callback=self._ask_timeout),
                ],
            },
            fallbacks=[
                CommandHandler(command='cancel', callback=self._ask_cancel)
            ],
            allow_reentry=True,
            conversation_timeout=datetime.timedelta(hours=1),
        ))

    def run(self):
        self.updater.start_polling()
        self.updater.idle()

    @dbconnect
    def _start(self, update: Update, context: CallbackContext, session: Session):
        bot_user = utils.get_bot_user_by_id(db=session, telegram_id=update.effective_user.id)
        if bot_user is not None and bot_user.state == BotUserState.active:
            message = f'Already authorized: `@{bot_user.staff_login}`'
        else:
            username = update.effective_user.username
            staff_info = staff_client.get_by_telegram(usernames=[username])
            if username in staff_info:
                bot_user = utils.get_or_create_bot_user(
                    db=session,
                    staff_info=staff_info[username],
                    username=username,
                    telegram_id=update.effective_user.id,
                )
                message = f'Hello, `@{bot_user.staff_login}`'

                email = bot_user.staff_login + '@yandex-team.ru'
                imported_subs = find_imported_subs(email=email, session=session)
                if imported_subs:
                    take_ownership(subs=imported_subs, bot_user=bot_user)
                    message += '. Found subscriptions from Stackoverflow:\n{}'.format(
                        f'`{email}`: *{", ".join({sub.tag for sub in imported_subs})}*'
                    )
            else:
                message = settings.NO_AUTH_MESSAGE

        context.bot.send_message(
            chat_id=update.effective_chat.id,
            text=message,
            parse_mode=ParseMode.MARKDOWN,
        )

    @auth_handler
    def _list_subscriptions(self, update: Update, session: Session, bot_user: BotUser):
        default_message_chat = 'No active subscriptions found for {}'.format(
            utils.get_chat_name(chat=update.effective_chat)
        )
        default_message_email = default_message_chat
        message_chat = ''
        message_email = ''
        has_email = False
        has_chat = False
        for title, model in (
            ('Subscriptions', Subscription),
            ('Unanswered subscriptions', SubscriptionUnanswered),
        ):
            chat_subscriptions = utils.query_subscriptions_by_chat(
                db=session, chat_id=update.effective_chat.id, model=model,
            ).all()
            if chat_subscriptions:
                has_chat = True
                message_chat += '{}{} of `{}`:\n{}'.format(
                    '\n' if message_chat else '',
                    title,
                    utils.get_chat_name(chat=update.effective_chat),
                    '\n'.join(f'*{sub.tag}*' for sub in chat_subscriptions)
                )

        if update.effective_chat.type != ChatType.private:
            return message_chat or default_message_chat

        for title, model in (
            ('subscriptions', Subscription),
            ('unanswered subscriptions', SubscriptionUnanswered),
        ):
            email_subscriptions = utils.query_email_subscriptions(
                db=session, author_id=bot_user.id, model=model
            ).all()
            email_to_tags = defaultdict(list)
            for sub in email_subscriptions:
                email_to_tags[sub.email.address].append(sub.tag)

            if email_subscriptions:
                has_email = True
                message_email += '{}`{}` has email {}:\n{}'.format(
                    '\n' if message_email else '',
                    bot_user.username,
                    title,
                    '\n'.join(f'`{email}`: *{", ".join(tags)}*' for email, tags in email_to_tags.items())
                )

        if not has_chat:
            return message_email or default_message_email
        if not has_email:
            return message_chat or default_message_chat

        return message_chat + '\nAlso, ' + message_email

    @auth_handler
    def _unsubscribe_all(self, update: Update, session: Session, bot_user: BotUser):
        default_message_chat = 'Nothing to unsubscribe from'
        default_message_email = default_message_chat

        message_chat = ''
        message_email = ''
        has_email = False
        has_chat = False

        for title, model in (('', Subscription), (' unanswered', SubscriptionUnanswered)):
            to_delete_subs = list()
            chat_subscriptions = utils.query_subscriptions_by_chat(
                db=session, chat_id=update.effective_chat.id,
                model=model,
            ).all()
            if chat_subscriptions:
                has_chat = True
                chat_tags = [sub.tag for sub in chat_subscriptions]
                to_delete_subs.extend([sub.id for sub in chat_subscriptions])

                logger.info(f'Unsubscribe {update.effective_chat.id} from {chat_tags} by {bot_user.id}')
                message_chat += '{}Successfully unsubscribed {} from{} tags:\n{}'.format(
                    '\n' if message_chat else '',
                    utils.get_chat_name(chat=update.effective_chat),
                    title,
                    '\n'.join(f'*{tag}*' for tag in chat_tags)
                )
                utils.delete_subscriptions(
                    db=session, subscription_ids=to_delete_subs, model=model
                )

        if update.effective_chat.type != ChatType.private:
            return message_chat or default_message_chat

        for title, model in (('', Subscription), (' unanswered', SubscriptionUnanswered)):
            to_delete_subs = []
            email_subscriptions = utils.query_email_subscriptions(
                db=session, author_id=bot_user.id,
                model=model,
            ).all()
            email_to_tags = defaultdict(list)
            for sub in email_subscriptions:
                to_delete_subs.append(sub.id)
                email_to_tags[sub.email.address].append(sub.tag)

            if email_subscriptions:
                has_email = True
                logger.info(f'Unsubscribe emails from tags: {email_to_tags} by {bot_user.id}')
                message_email += '{}Unsubscribed emails from{} tags:\n{}'.format(
                    '\n' if message_email else '',
                    title,
                    '\n'.join(f'`{email}`: *{", ".join(tags)}*' for email, tags in email_to_tags.items())
                )

                utils.delete_subscriptions(db=session, subscription_ids=to_delete_subs, model=model)
        if not has_chat:
            return message_email or default_message_email
        if not has_email:
            return message_chat or default_message_chat

        return message_chat + '\nAlso, ' + message_email

    @auth_handler
    def _unsubscribe(self, update: Update, context: CallbackContext, session: Session, bot_user: BotUser):
        if not context.args or len(context.args) > 2:
            return 'You should pass exactly one tag'

        tag = context.args[0]

        model = Subscription
        if len(context.args) == 2:
            if context.args[1] == '--digest':
                model = SubscriptionUnanswered
            else:
                return 'You should pass exactly one tag'

        subscription = utils.get_subscription(
            db=session, tag=tag,
            chat_id=update.effective_chat.id,
            model=model,
        )

        chat_name = utils.get_chat_name(chat=update.effective_chat)
        if not subscription:
            return f'`{chat_name}` not subscribed to tag: *{tag}*'

        utils.delete_subscriptions(
            db=session, subscription_ids=[subscription.id],
            model=model,
        )
        logger.info(f'Unsubscribe {update.effective_chat.id} from {tag} by {bot_user.id}')

        return f'Successfully unsubscribed `{chat_name}` from tag: *{tag}*'

    @auth_handler
    def _subscribe(self, update: Update, context: CallbackContext, session: Session, bot_user: BotUser):
        if not context.args or len(context.args) > 2:
            return 'You should pass exactly one tag'

        tag = context.args[0]
        digest = False
        model = Subscription
        if len(context.args) == 2:
            if context.args[1] == '--digest':
                digest = True
                model = SubscriptionUnanswered
            else:
                'You should pass exactly one tag'

        already_subscribed = utils.get_subscription(
            db=session, tag=tag, chat_id=update.effective_chat.id,
            model=model,
        )
        chat_name = utils.get_chat_name(chat=update.effective_chat)
        if already_subscribed:
            return f'`{chat_name}` already subscribed to tag: *{tag}*'

        tag_exists = startrek_client.check_if_tag_exists(tag=tag)
        if not tag_exists:
            return f'No such tag: *{tag}*'

        chat = utils.get_or_create_chat(db=session, chat_data=update.effective_chat, author_id=bot_user.id)
        utils.create_subscription(
            db=session, tag=tag, chat=chat,
            model=model,
        )
        logger.info(f'Subscribe {chat.id} for {tag} by {bot_user.id}, {digest}')
        message = f'Successfully subscribed `{chat_name}` to tag: *{tag}*'
        if digest:
            message = f'Successfully subscribed `{chat_name}` for weekly digest to tag: *{tag}*'
        return message

    @auth_handler
    def _subscribe_email(self, update: Update, context: CallbackContext, session: Session, bot_user: BotUser):
        if update.effective_chat.type != ChatType.private:
            return 'This command restricted to private chats only'

        if not context.args or 2 > len(context.args) > 3:
            return 'You should pass exactly one tag and email address'

        tag, email_addr = context.args[:2]
        if not email_addr.endswith('@yandex-team.ru'):
            tag, email_addr = email_addr, tag
        if not email_addr.endswith('@yandex-team.ru'):
            return 'Email domain restricted to *yandex-team.ru*'

        digest = False
        model = Subscription
        if len(context.args) == 3:
            if context.args[2] == '--digest':
                digest = True
                model = SubscriptionUnanswered
            else:
                return 'You should pass exactly one tag and email address'

        subscription = utils.get_email_subscription(
            db=session, tag=tag, email_address=email_addr,
            model=model,
        )
        if subscription:
            message = f'This email already subscribed to tag: *{tag}*'
            if subscription.email.author_id != bot_user.id:
                message = message + f' by `@{subscription.email.author.staff_login}`'
            return message

        tag_exists = startrek_client.check_if_tag_exists(tag=tag)
        if not tag_exists:
            return f'No such tag: *{tag}*'

        email = utils.get_or_create_email(db=session, email_address=email_addr, author_id=bot_user.id)
        utils.create_email_subscription(
            db=session, tag=tag, email=email,
            model=model,
        )

        logger.info(f'Subscribe email {email.id} for {tag} by {bot_user.id}, {digest}')
        message = f'Successfully subscribed `{email_addr}` to tag: *{tag}*'
        if digest:
            message = f'Successfully subscribed `{email_addr}` for weekly digest to tag: *{tag}*'
        return message

    @auth_handler
    def _unsubscribe_email(self, update: Update, context: CallbackContext, session: Session, bot_user: BotUser):
        if update.effective_chat.type != ChatType.private:
            return 'This command restricted to private chats only'

        if not context.args or 2 > len(context.args) > 3:
            return 'You should pass exactly one tag and email address'

        model = Subscription
        if len(context.args) == 3:
            if context.args[2] == '--digest':
                model = SubscriptionUnanswered
            else:
                return 'You should pass exactly one tag and email address'

        tag, email_addr = context.args[:2]
        if not email_addr.endswith('@yandex-team.ru'):
            tag, email_addr = email_addr, tag
        if not email_addr.endswith('@yandex-team.ru'):
            return 'Email domain restricted to *yandex-team.ru*'

        subscription = utils.get_email_subscription(
            db=session, tag=tag, email_address=email_addr,
            model=model,
        )
        if subscription is None:
            return f'`{email_addr}` not subscribed to tag: *{tag}*'
        if subscription.email.author_id != bot_user.id:
            return f'Only `@{subscription.email.author.staff_login}` can unsubscribe email from tag'

        utils.delete_subscriptions(
            db=session, subscription_ids=[subscription.id],
            model=model,
        )
        logger.info(f'Unsubscribe email {subscription.email.id} from {tag} by {bot_user.id}')
        return f'Successfully unsubscribed `{email_addr}` from tag: *{tag}*'

    @auth_handler
    def _search(self, context: CallbackContext):
        search_str = ""
        if context.args:
            search_str = ' '.join(context.args)
        if not search_str:
            return 'Empty search parameter.\n'

        questions_info = startrek_client.search_questions(param=search_str, limit=5)
        if not questions_info:
            return f'We couldn\'t find anything for *{search_str}*\nTry different or less specific keywords.'

        return 'Results for *{search_str}*:\n{results}\n\n{results_link}'.format(
            search_str=search_str,
            results='\n'.join([
                '[{}]({})'.format(
                    html.unescape(question.summary), startrek_client.get_link(question),
                ) for question in questions_info
            ]),
            results_link='[All results]({}/questions?text={})'.format(
                settings.STACKOVERFLOW_HOST, urllib.parse.quote(search_str.encode('utf-8'), safe='')
            )
        )

    def _clear_ask_context(self, context: CallbackContext):
        if 'tags' in context.user_data:
            del context.user_data['tags']
        if 'title' in context.user_data:
            del context.user_data['title']
        if 'body' in context.user_data:
            del context.user_data['body']

    @auth_handler
    def _ask_start(self, update: Update, context: CallbackContext):
        cancel_msg = '\nType `/cancel` anytime to cancel question creation'
        if update.effective_message.reply_to_message:
            message = update.effective_message.reply_to_message
            text = []
            if message.caption_markdown_v2_urled:
                text.append(message.caption_markdown_v2_urled)
            if message.text_markdown_v2_urled:
                text.append(message.text_markdown_v2_urled)
            context.user_data['body'] = '\n'.join(text)

            if not context.args:
                return AskStates.REPLY_TITLE,  'Please, enter **question title**.' + cancel_msg

            context.user_data['title'] = ' '.join(context.args)
            return AskStates.REPLY_TAGS,  'Please, enter **question tags**.' + cancel_msg

        if context.args:
            context.user_data['title'] = ' '.join(context.args)
        return AskStates.REPLY_BODY,  'Please, enter **question body**.' + cancel_msg

    @auth_handler
    def _ask_reply_body(self, update: Update, context: CallbackContext):
        text = []
        if update.effective_message.caption_markdown:
            text.append(update.effective_message.caption_markdown)
        if update.effective_message.text_markdown:
            text.append(update.effective_message.text_markdown)

        context.user_data['body'] = unescape_body(text='\n'.join(text))

        if 'title' in context.user_data and context.user_data['title'] is not None:
            return AskStates.REPLY_TAGS,  'Please, enter **question tags**.'
        return AskStates.REPLY_TITLE,  'Please, enter **question title**.'

    @auth_handler
    def _ask_reply_title(self, update: Update, context: CallbackContext):
        context.user_data['title'] = update.effective_message.text
        return AskStates.REPLY_TAGS, 'Please, enter **question tags**.'

    @auth_handler
    def _ask_reply_tags(self, update: Update, context: CallbackContext):
        text = update.effective_message.text
        tags = set(text.replace(',', ' ').replace(';', ' ').split())
        context.user_data['tags'] = tags

        existing_tags = startrek_client.get_existing_tags(tags=tags)
        if existing_tags != tags:
            message = 'Not all tags exist: {}. {}'.format(
                ', '.join(f'*{tag}*' for tag in tags.difference(existing_tags)),
                'Please enter the tags again, or, if you really want to create them, type `/confirm`',
            )
            return AskStates.CONFIRM_TAGS, message

        title = context.user_data['title']
        body = context.user_data['body']
        tags = ', '.join(f'`{tag}`' for tag in tags)

        message = f'*{title}*\n{tags}\n\n{body}\n\nAll seems OK, `/done` when you\'re ready.'
        return AskStates.PREVIEW_QUESTION, message

    @auth_handler
    def _ask_confirm_tags(self, context: CallbackContext):
        title = context.user_data['title']
        body = context.user_data['body']
        tags = ', '.join(f'`{tag}`' for tag in context.user_data['tags'])

        message = f'*{title}*\n{tags}\n\n{body}\n\nAll seems OK, `/done` when you\'re ready.'
        return AskStates.PREVIEW_QUESTION, message

    @auth_handler
    def _ask_finish(self, context: CallbackContext, bot_user: BotUser):
        title = context.user_data['title']
        tags = context.user_data['tags']
        body = context.user_data['body']

        issue = startrek_client.create_question(
            title=title, tags=tags, body=body + f'\n\nАвтор вопроса @{bot_user.staff_login}'
        )
        if issue is None:
            message = 'Question creation error.'
        else:
            link = startrek_client.get_link(issue)
            message = f'Successfully created question:\n[{issue.key}: {title}]({link})'

        self._clear_ask_context(context=context)
        return ConversationHandler.END, message

    @auth_handler
    def _ask_timeout(self, context: CallbackContext):
        self._clear_ask_context(context=context)
        return ConversationHandler.END, 'Question creation canceled due to timeout'

    @auth_handler
    def _ask_cancel(self, context: CallbackContext):
        self._clear_ask_context(context=context)
        return ConversationHandler.END, 'Question creation canceled'


def main():
    parser = argparse.ArgumentParser(add_help=True)
    parser.add_argument('--telegram-token', required=False)
    args = parser.parse_args()

    bot = BotHandler(token=args.telegram_token)
    bot.run()
