import itertools
import logging
from typing import Iterable

import smtplib
import html

from datetime import date
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

import yt.wrapper as yt
from telegram import ParseMode
from telegram.error import (
    TelegramError,
    Unauthorized,
    ChatMigrated,
)
from collections import defaultdict
from sqlalchemy.orm import Session, joinedload
from sqlalchemy import or_, and_, tuple_

from stackbot.so_stat_task import (
    create_yt_table,
    create_latest_link,
    save_from_so_to_yt,
    __yt_a_path__,
    __yt_q_path__,
    __yt_u_path__,
    __yt_u_schema__,
    __yt_a_schema__,
    __yt_q_schema__,
    __yt_cluster__,
)
from stackbot.enums import (
    BotUserState,
    EventType,
    EventState,
    ChatType,
    SubscriptionEventState,
    SubscriptionType,
    LogbrokerEventState,
)
from stackbot.db import (
    dbconnect,
    Event,
    LogbrokerEvent,
    Subscription,
    SubscriptionToEvent,
    BotUser,
    StackUser,
    Chat,
    Email,
    SubscriptionUnanswered,
)
from stackbot.logic.utils import (
    get_chunks,
)
from stackbot.main import get_bot
from stackbot.celery_app import app
from stackbot.config import settings
from stackbot.logic.clients.stack import StackClient
from stackbot.logic.clients.startrek import StartrekClient
from stackbot.logic.clients.staff import staff_client
from stackbot.logic.clients.achievery import achievery_client
from stackbot.logic.logbroker import (
    LogbrokerClient,
    Message,
)


logger = logging.getLogger(__name__)


question_posted_email_template = '''
<p>
    New question with tag{s} {tags} by <a href="https://staff.yandex-team.ru/{author}">@{author}</a>
</p>
<h2>
    {q_title}
</h2>
<p>
    {q_body}
</p>
<br>
<a href="{q_link}">Go to question</a>
<p>
    You can manage your subscriptions via telegram bot
    <a href="https://t.me/yastackbot">@YaStackBot</a>
</p>
'''


unanswered_questions_email_template = '''
<p>
    New question with tag{s} {tags} by <a href="https://staff.yandex-team.ru/{author}">@{author}</a>
</p>
<h2>
    {q_title}
</h2>
<p>
    {q_body}
</p>
<br>
<a href="{q_link}">Go to question</a>
<p>
    You can manage your subscriptions via telegram bot
    <a href="https://t.me/yastackbot">@YaStackBot</a>
</p>
'''


@app.task
@dbconnect
def synchronize_bot_users(session: Session) -> None:
    active_bot_users = session.query(BotUser).filter(
        BotUser.state == BotUserState.active,
        BotUser.staff_login != settings.ROBOT_LOGIN,
    ).all()
    to_deprive = list()
    for chunk in get_chunks(active_bot_users):
        staff_info = staff_client.get_by_telegram(
            usernames=[bot_user.username for bot_user in chunk]
        )

        for bot_user in chunk:
            if bot_user.username not in staff_info:
                to_deprive.append(bot_user.id)

    session.query(BotUser).filter(BotUser.id.in_(to_deprive)).update(
        {BotUser.state: BotUserState.deprived},
        synchronize_session=False,
    )


@app.task
@dbconnect
def sync_events(session: Session) -> None:
    logger.info('Starting syncing events')
    events = StackClient().get_events()
    question_posted_event_ids = [
        event['event_id']
        for event in events
        if event['event_type'] == EventType.question_posted
    ]
    if question_posted_event_ids:
        already_created = session.query(Event.event_id).filter(
            Event.event_id.in_(question_posted_event_ids),
            Event.type == EventType.question_posted,
        )
        already_created_ids = {
            event for event, in already_created
        }
        session.bulk_insert_mappings(
            Event,
            [
                {
                    'event_id': event_id,
                    'type': EventType.question_posted,
                    'state': EventState.new,
                }
                for event_id in question_posted_event_ids
                if event_id not in already_created_ids
            ]
        )
    logger.info('Finish syncing events')


@app.task
@dbconnect
def process_question_posted_event(session: Session) -> None:
    logger.info('Starting processing question_posted')

    #  получаем данные вопросов из апи
    question_posted_events = session.query(Event).filter(
        type=EventType.question_posted,
        state=EventState.new,
    ).all()
    questions = StackClient().get_questions(
        ids=[event.event_id for event in question_posted_events]
    )
    question_by_tags = {
        question['question_id']: question['tags']
        for question in questions
    }
    all_tags = set()
    for tags in question_by_tags.values():
        all_tags.update(tags)

    #  получаем активные подписки из базы
    tags_subscription = (
        session.query(Subscription)
        .join(Chat, Chat.id == Subscription.chat_id, isouter=True)
        .join(Email, Email.id == Subscription.email_id, isouter=True)
        .join(
            BotUser,
            or_(
                and_(
                    Subscription.type == SubscriptionType.chat,
                    BotUser.id == Chat.author_id
                ),
                and_(
                    Subscription.type == SubscriptionType.email,
                    BotUser.id == Email.author_id
                )
            )
        )
        .filter(
            Subscription.tag.in_(all_tags),
            or_(
                BotUser.state == BotUserState.active,
                and_(
                    Subscription.type == SubscriptionType.chat,
                    Chat.chat_type.in_(ChatType.group_types())
                )
            )
        )
    )

    tags_map = defaultdict(list)
    for tag_subscription in tags_subscription:
        tags_map[tag_subscription.tag].append(tag_subscription)

    already_created = session.query(SubscriptionToEvent).filter(
        SubscriptionToEvent.event_id.in_(
            [event.id for event in question_posted_events]
        )
    )
    already_created_map = defaultdict(set)
    for item in already_created:
        already_created_map[item.event_id].add(item.subscription_id)

    #  создаем недостающие
    to_create = []
    for event in question_posted_events:
        question_tags = question_by_tags[event.event_id]
        for tag in question_tags:
            subscriptions = tags_map[tag]
            for subscription in subscriptions:
                if subscription.id not in already_created_map[event.id]:
                    to_create.append(
                        {
                            'subscription_id': subscription.id,
                            'event_id': event.id,
                            'state': SubscriptionEventState.new,
                        }
                    )
    if to_create:
        session.bulk_insert_mappings(
            SubscriptionToEvent,
            to_create,
        )
    question_posted_events.update(
        {Event.state: EventState.done},
        synchronize_session=False,
    )
    logger.info('Finish processing question_posted')


@app.task
@dbconnect
def process_recent_active_questions(session: Session) -> None:
    logger.info('Starting processing recent active questions')
    issues = StartrekClient().get_recent_active_questions(gap_in_mins=settings.TASK_QUESTIONS_SYNC_GAP_IN_MINS)
    key_to_tags = {int(issue.key.partition('-')[2]): issue.tags for issue in issues}
    all_tags = set()
    for _, tags in key_to_tags.items():
        all_tags.update(tags)

    subscriptions = (
        session.query(Subscription)
        .join(Chat, Chat.id == Subscription.chat_id, isouter=True)
        .join(Email, Email.id == Subscription.email_id, isouter=True)
        .join(
            BotUser,
            or_(
                and_(
                    Subscription.type == SubscriptionType.chat,
                    BotUser.id == Chat.author_id
                ),
                and_(
                    Subscription.type == SubscriptionType.email,
                    BotUser.id == Email.author_id
                )
            )
        )
        .filter(
            Subscription.tag.in_(all_tags),
            or_(
                BotUser.state == BotUserState.active,
                and_(
                    Subscription.type == SubscriptionType.chat,
                    Chat.chat_type.in_(ChatType.group_types())
                )
            )
        )
    )
    tag_to_subscriptions = defaultdict(set)
    subscription_ids = set()
    for sub in subscriptions:
        tag_to_subscriptions[sub.tag].add(sub.id)
        subscription_ids.add(sub.id)

    sub_to_events = (
        session.query(SubscriptionToEvent)
        .join(Event, Event.id == SubscriptionToEvent.event_id)
        .filter(SubscriptionToEvent.subscription_id.in_(subscription_ids))
        .filter(Event.event_id.in_(key_to_tags))
        .options(joinedload(SubscriptionToEvent.event))
    )
    subscriptions_map = defaultdict(set)
    for sub_to_event in sub_to_events:
        subscriptions_map[sub_to_event.subscription_id].add(sub_to_event.event.event_id)

    existing_events = (
        session.query(Event)
        .filter(Event.type == EventType.question_posted)
        .filter(Event.event_id.in_(key_to_tags))
    )
    existing_event_ids = {event.event_id for event in existing_events}
    to_create_events = list()
    new_question_events = set()
    for issue_key, tags in key_to_tags.items():
        if issue_key in existing_event_ids:
            continue
        for tag in tags:
            if tag not in tag_to_subscriptions:
                continue
            for sub_id in tag_to_subscriptions[tag]:
                if sub_id not in subscriptions_map or issue_key not in subscriptions_map[sub_id]:
                    if issue_key not in new_question_events:
                        new_question_events.add(issue_key)
                        to_create_events.append(Event(
                            event_id=issue_key,
                            type=EventType.question_posted,
                            state=EventState.new,
                        ))

    all_recent_events = existing_events.all()

    if to_create_events:
        session.bulk_save_objects(to_create_events, return_defaults=True)
        session.flush()
    all_recent_events.extend(to_create_events)

    to_create = list()
    for event in all_recent_events:
        question_tags = key_to_tags[event.event_id]
        for tag in question_tags:
            if tag not in tag_to_subscriptions:
                continue
            for sub_id in tag_to_subscriptions[tag]:
                if sub_id not in subscriptions_map or event.event_id not in subscriptions_map[sub_id]:
                    to_create.append({
                        'subscription_id': sub_id,
                        'event_id': event.id,
                        'state': SubscriptionEventState.new,
                    })
    if to_create:
        session.bulk_insert_mappings(
            SubscriptionToEvent,
            to_create,
        )
    if all_recent_events:
        session.query(Event).filter(
            Event.id.in_({event.id for event in all_recent_events})
        ).update(
            {Event.state: EventState.done},
            synchronize_session=False,
        )
    logger.info('Finishing processing recent active questions')


def _group_sub_to_events(source: Iterable[SubscriptionToEvent]) -> tuple[defaultdict[int, set], defaultdict[int, set]]:
    chat_map, email_map = defaultdict(set), defaultdict(set)
    for sub_to_event in source:
        sub, event = sub_to_event.subscription, sub_to_event.event
        if sub.type == SubscriptionType.chat:
            chat_map[sub.chat_id].add(event.event_id)
        elif sub.type == SubscriptionType.email:
            email_map[sub.email_id].add(event.event_id)
    return chat_map, email_map


def _handle_chat_migration(session: Session, chat_id: int, exc: ChatMigrated) -> None:
    old_chat = session.query(Chat).get(chat_id)
    if not session.query(Chat).get(exc.new_chat_id):
        # Если new_chat_id еще не существует, то создаем его
        session.add(Chat(
            id=exc.new_chat_id,
            title=old_chat.title,
            author_id=old_chat.author_id,
            chat_type=ChatType.supergroup,
        ))
    # Мигрируем все Subscription с chat_id на new_chat_id
    for model in (Subscription, SubscriptionUnanswered):
        session.query(model).filter(
            model.type == SubscriptionType.chat,
            model.chat_id == chat_id,
        ).update({
            model.chat_id: exc.new_chat_id,
        }, synchronize_session=False)

    session.delete(old_chat)

    logger.info('Chat {} migrated to {}'.format(chat_id, exc.new_chat_id))


@app.task
@dbconnect
def send_question_posted_event(session: Session) -> None:
    logger.info('Starting processing question_posted')

    telegram_template = 'New question on tag{s} *{tags}:*\n[{q_title}]({q_link})'
    email_subject_template = 'Q[{tags}]: {q_title}'

    to_process = session.query(SubscriptionToEvent).join(
        Event, Event.id == SubscriptionToEvent.event_id
    ).filter(
        SubscriptionToEvent.state == SubscriptionEventState.new,
        Event.type == EventType.question_posted,
    ).options(
        joinedload(SubscriptionToEvent.subscription).joinedload(Subscription.email),
        joinedload(SubscriptionToEvent.subscription).joinedload(Subscription.chat),
        joinedload(SubscriptionToEvent.event),
    ).all()

    issue_keys = [sub_to_event.event.event_id for sub_to_event in to_process]
    if not issue_keys:
        logger.info('Finish processing question_posted. Nothing to send.')
        return
    questions_info = StartrekClient().get_questions_info(keys=issue_keys)

    processed = (
        session.query(SubscriptionToEvent)
        .join(Event, Event.id == SubscriptionToEvent.event_id)
        .join(Subscription, Subscription.id == SubscriptionToEvent.subscription_id)
        .filter(
            SubscriptionToEvent.state == SubscriptionEventState.send,
            Event.type == EventType.question_posted,
            Event.event_id.in_(issue_keys),
        )
        .options(
            joinedload(SubscriptionToEvent.subscription).joinedload(Subscription.email),
            joinedload(SubscriptionToEvent.subscription).joinedload(Subscription.chat),
            joinedload(SubscriptionToEvent.event),
        ).all()
    )

    chat_to_question, email_to_question = _group_sub_to_events(to_process)
    processed_chat_to_question, processed_email_to_question = _group_sub_to_events(processed)

    for chat_id in chat_to_question:
        if chat_id in processed_chat_to_question:
            chat_to_question[chat_id].difference_update(processed_chat_to_question[chat_id])
    for email_id in email_to_question:
        if email_id in processed_email_to_question:
            email_to_question[email_id].difference_update(processed_email_to_question[email_id])

    to_reject = list()
    grouped_email, grouped_chat = defaultdict(lambda: dict()), defaultdict(lambda: dict())
    for subscription_event in to_process:
        sub = subscription_event.subscription
        issue_key = subscription_event.event.event_id
        if issue_key not in questions_info:
            logger.error('Question id not found: {}'.format(issue_key))
            to_reject.append(subscription_event.id)
            continue

        question_info = questions_info[issue_key]
        title, link = html.unescape(question_info.summary), StartrekClient.get_link(question_info)

        if sub.type == SubscriptionType.chat and issue_key in chat_to_question[sub.chat_id]:
            if issue_key not in grouped_chat[sub.chat_id]:
                grouped_chat[sub.chat_id][issue_key] = {
                    '_ids': [], 'tags': [],
                    '_author_id': sub.chat.author_id,
                    'title': title, 'link': link,
                }
            grouped_chat[sub.chat_id][issue_key]['_ids'].append(subscription_event.id)
            grouped_chat[sub.chat_id][issue_key]['tags'].append(sub.tag)
        elif sub.type == SubscriptionType.email and issue_key in email_to_question[sub.email_id]:
            if issue_key not in grouped_email[sub.email.address]:
                grouped_email[sub.email.address][issue_key] = {
                    '_ids': [], 'tags': [],
                    '_author_id': sub.email.author_id,
                    'title': title, 'link': link,
                    'body': question_info.description,
                }
            grouped_email[sub.email.address][issue_key]['_ids'].append(subscription_event.id)
            grouped_email[sub.email.address][issue_key]['tags'].append(sub.tag)
        else:
            to_reject.append(subscription_event.id)

    to_delete_sub_to_events = list()
    to_deprive_bot_users = list()
    sent = []
    bot = get_bot(token=settings.TELEGRAM_TOKEN)

    for chat_id, messages in grouped_chat.items():
        for _, msg in messages.items():
            tags = msg['tags']
            text = telegram_template.format(
                s='s' if len(tags) > 1 else '',
                tags=', '.join(tags),
                q_link=msg['link'],
                q_title=msg['title'],
            )
            try:
                bot.send_message(chat_id=chat_id, text=text, parse_mode=ParseMode.MARKDOWN)
                sent.extend(msg['_ids'])
            except Unauthorized as exc:
                author_id = msg['_author_id']
                to_delete_sub_to_events.extend(msg['_ids'])
                to_deprive_bot_users.append(author_id)
                logger.info('User: {} blocked bot: {}'.format(author_id, exc))
            except ChatMigrated as exc:
                _handle_chat_migration(session=session, chat_id=chat_id, exc=exc)
                try:
                    bot.send_message(chat_id=exc.new_chat_id, text=text, parse_mode=ParseMode.MARKDOWN)
                except TelegramError as exc:
                    logger.exception('Failed to send telegram message: {}'.format(exc))
            except TelegramError as exc:
                logger.exception('Failed to send telegram message: {}'.format(exc))

    smtp = smtplib.SMTP(host=settings.EMAIL_HOST, port=settings.EMAIL_PORT)
    for email_address, messages in grouped_email.items():
        for _, msg in messages.items():
            tags = msg['tags']
            email_subject = email_subject_template.format(
                tags=', '.join(tags),
                q_title=msg['title'],
            )
            body, _, author = msg['body'].partition('\n\nАвтор вопроса @')
            text = question_posted_email_template.format(
                s='s' if len(tags) > 1 else '',
                tags=', '.join(tags),
                q_title=msg['title'],
                q_body=body,
                q_link=msg['link'],
                author=author,
            )

            message = MIMEMultipart()
            message['Subject'] = email_subject
            message['From'] = settings.EMAIL_ADDR
            message['To'] = email_address
            message.attach(MIMEText(text, 'html'))

            try:
                smtp.send_message(message)
                sent.extend(msg['_ids'])
            except smtplib.SMTPException as exc:
                logger.exception('Failed to send email message: {}'.format(exc))

    smtp.quit()
    if sent:
        session.query(SubscriptionToEvent).filter(
            SubscriptionToEvent.id.in_(sent)
        ).update(
            {SubscriptionToEvent.state: SubscriptionEventState.send},
            synchronize_session=False,
        )

    if to_reject:
        session.query(SubscriptionToEvent).filter(
            SubscriptionToEvent.id.in_(to_reject)
        ).update({
            SubscriptionToEvent.state: SubscriptionEventState.rejected,
        }, synchronize_session=False)

    if to_delete_sub_to_events:
        session.query(SubscriptionToEvent).filter(
            SubscriptionToEvent.id.in_(to_delete_sub_to_events)
        ).delete(synchronize_session=False)

        session.query(BotUser).filter(BotUser.id.in_(to_deprive_bot_users)).update({
            BotUser.state: BotUserState.deprived,
        }, synchronize_session=False)

    logger.info('Finish processing question_posted. Sent {} out of {}'.format(len(sent), len(issue_keys)))


@app.task
@dbconnect
def send_logbroker_updates(session: Session) -> None:
    logger.info('Logbroker updates are disabled')
    return

    logger.info('Starting processing send_logbroker_updates')

    base_url = f'https://stackoverflow.yandex-team.ru/api/{StackClient().API_VERSION}'
    ev_type_to_url = {
        'question_posted': '{base_url}/questions/{ev_id}?order=desc&sort=activity',
        'answer_posted': '{base_url}/answers/{ev_id}?order=desc&sort=activity',
        'comment_posted': '{base_url}/comments/{ev_id}?order=desc&sort=creation',
        'post_edited': '{base_url}/posts/{ev_id}?order=desc&sort=activity',
    }

    not_processed_events = session.query(LogbrokerEvent).filter(
        LogbrokerEvent.state == LogbrokerEventState.new,
    )

    events = StackClient().get_events()
    event_keys = {(ev['event_id'], ev['creation_date']) for ev in events}

    processed_events = session.query(LogbrokerEvent).filter(
        tuple_(LogbrokerEvent.event_id, LogbrokerEvent.creation_date).in_(event_keys),
        LogbrokerEvent.state == LogbrokerEventState.sent,
    )
    existing_keys = {
        ev.get_key() for ev in itertools.chain(not_processed_events, processed_events)
    }

    to_create = list()
    for ev in events:
        if ev['event_type'] not in ev_type_to_url:
            continue
        if (ev['event_id'], ev['creation_date']) in existing_keys:
            continue

        to_create.append(LogbrokerEvent(
            creation_date=ev['creation_date'],
            event_id=ev['event_id'],
            type=ev['event_type'],
            link=ev['link'],
        ))

    if to_create:
        session.bulk_save_objects(to_create)

    to_send_events = session.query(LogbrokerEvent).filter(
        LogbrokerEvent.state == LogbrokerEventState.new,
    ).all()
    if not to_send_events:
        logger.info('Finish processing send_logbroker_updates. No new updates.')
        return

    messages = list()
    for ev in to_send_events:
        messages.append(
            Message(
                _id=ev.id,
                content={
                    'id': ev.event_id,
                    'apiurl': ev_type_to_url[ev.type].format(base_url=base_url, ev_id=ev.event_id),
                    'url': ev.link,
                    'fetch': True,
                },
            )
        )

    logbroker_client = LogbrokerClient(
        endpoint=settings.LOGBROKER_ENDPOINT,
        topic=settings.LOGBROKER_TOPIC,
        producer=settings.LOGBROKER_PRODUCER,
    )
    completed_event_ids = logbroker_client.write_messages(messages=messages)
    logbroker_client.shutdown()

    session.query(LogbrokerEvent).filter(
        LogbrokerEvent.id.in_(completed_event_ids)
    ).update({
        LogbrokerEvent.state: LogbrokerEventState.sent,
    }, synchronize_session=False)

    logger.info('Finish processing send_logbroker_updates. Sent {} out of {}'.format(
        len(completed_event_ids), len(messages),
    ))


@app.task
@dbconnect
def send_unanswered_digest(session: Session):
    logger.info('Sending unanswered questions digest')
    subscriptions = (
        session.query(SubscriptionUnanswered)
        .join(Chat, Chat.id == SubscriptionUnanswered.chat_id, isouter=True)
        .join(Email, Email.id == SubscriptionUnanswered.email_id, isouter=True)
        .join(
            BotUser,
            or_(
                and_(
                    SubscriptionUnanswered.type == SubscriptionType.chat,
                    BotUser.id == Chat.author_id
                ),
                and_(
                    SubscriptionUnanswered.type == SubscriptionType.email,
                    BotUser.id == Email.author_id
                )
            )
        )
        .filter(
            or_(
                BotUser.state == BotUserState.active,
                and_(
                    SubscriptionUnanswered.type == SubscriptionType.chat,
                    Chat.chat_type.in_(ChatType.group_types())
                )
        )
    )).all()
    tags = {item.tag for item in subscriptions}
    tags_map = defaultdict(list)
    question_issues = StartrekClient().unanswered_questions_by_tags(tags=tags)
    for issue in question_issues.values():
        for tag in issue.tags:
            tags_map[tag].append(issue)

    if not tags_map:
        logger.info(f'No unanswered questions by {tags} found')
        return

    bot = get_bot(token=settings.TELEGRAM_TOKEN)
    smtp = smtplib.SMTP(host=settings.EMAIL_HOST, port=settings.EMAIL_PORT)

    for subscription in subscriptions:
        tag = subscription.tag
        issues = tags_map.get(tag)[:5]
        if not issues:
            continue
        if subscription.type == SubscriptionType.chat:
            message = 'Unanswered questions for *{tag}*:\n{results}'.format(
                tag=tag,
                results='\n'.join([
                    '[{}]({})'.format(
                        html.unescape(question.summary), StartrekClient.get_link(question),
                    ) for question in question_issues
                ]),
            )
            try:
                bot.send_message(chat_id=subscription.chat_id, text=message, parse_mode=ParseMode.MARKDOWN)
            except ChatMigrated as exc:
                _handle_chat_migration(session=session, chat_id=subscription.chat_id, exc=exc)
                try:
                    bot.send_message(chat_id=exc.new_chat_id, text=message, parse_mode=ParseMode.MARKDOWN)
                except TelegramError as exc:
                    logger.exception('Failed to send telegram message: {}'.format(exc))
            except TelegramError as exc:
                logger.exception('Failed to send telegram message: {}'.format(exc))
        else:
            subject = f'Unanswered questions for [{tag}]'

            text = '<br>'.join([
                '<a href="{}">{}</a>'.format(
                    StartrekClient.get_link(question), html.unescape(question.summary),
                ) for question in question_issues
            ])

            message = MIMEMultipart()
            message['Subject'] = subject
            message['From'] = settings.EMAIL_ADDR
            message['To'] = subscription.email.address
            message.attach(MIMEText(text, 'html'))

            try:
                smtp.send_message(message)
            except smtplib.SMTPException as exc:
                logger.exception('Failed to send email message: {}'.format(exc))
    smtp.quit()


@app.task
def upload_stat_to_yt():
    logger.info('Uploading to YT is disabled')
    return

    logger.info('Uploading to YT')
    yt_client = yt.YtClient(__yt_cluster__, config={'token': settings.YT_TOKEN})
    today = date.today().strftime("%Y-%m-%d")

    # Answers
    yt_a_today_path = '/'.join([__yt_a_path__, today])
    create_yt_table(yt_client, yt_a_today_path, __yt_a_schema__)
    save_from_so_to_yt(yt_client, 'all_answers', yt_a_today_path)
    create_latest_link(yt_client, __yt_a_path__, yt_a_today_path)

    # Questions
    yt_q_today_path = '/'.join([__yt_q_path__, today])
    create_yt_table(yt_client, yt_q_today_path, __yt_q_schema__)
    save_from_so_to_yt(yt_client, 'all_questions', yt_q_today_path)
    create_latest_link(yt_client, __yt_q_path__, yt_q_today_path)

    # Users
    yt_u_today_path = '/'.join([__yt_u_path__, today])
    create_yt_table(yt_client, yt_u_today_path, __yt_u_schema__)
    save_from_so_to_yt(yt_client, 'all_users', yt_u_today_path)
    create_latest_link(yt_client, __yt_u_path__, yt_u_today_path)

    logger.info('Finish uploading to YT')


@app.task
@dbconnect
def update_achievements(session: Session):
    logger.info('Starting update_achievements task')
    accepted_counts = StackClient().get_users_accepted_answer_counts()

    existing_users = session.query(StackUser).all()

    current_accepted_counts = dict()
    to_update = list()

    for user in existing_users:
        current_accepted_counts[user.display_name] = user.accepted_count
        new_accepted_count = accepted_counts.get(user.display_name, user.accepted_count)
        if new_accepted_count != user.accepted_count:
            to_update.append({'id': user.id, 'accepted_count': new_accepted_count})

    to_create = list()
    for user, accepted_count in accepted_counts.items():
        if user not in current_accepted_counts:
            to_create.append({'display_name': user, 'accepted_count': accepted_count})

    if to_update:
        session.bulk_update_mappings(StackUser, to_update)
    if to_create:
        session.bulk_insert_mappings(StackUser, to_create)

    new_achievement = dict()
    for user, new_accepted_count in accepted_counts.items():
        old_accepted_count = current_accepted_counts[user]
        for answer_count, achievement_id in settings.STAFF_ACHIEVEMENTS_X_ACCEPTED_ANSWERS:
            if old_accepted_count < answer_count <= new_accepted_count:
                new_achievement[user] = achievement_id

    achievery_client.update_staff_achievements(new_achievement)
    logger.info('Finish update_achievements task')
