# coding: utf-8
from __future__ import unicode_literals

import logging
import re
import socket
from urllib import urlencode

from django.conf import settings
from django.http import HttpResponse, HttpResponseServerError

from django_yauth.middleware import YandexAuthBackendMiddleware
from django_yauth.authentication_mechanisms.tvm.request import TvmServiceRequest
from django_yauth.user import YandexUser

from static_api.views.resource import _get_request_data

logger = logging.getLogger(__name__)


def check_host(host, port):
    sockfamily = socket.AF_INET6
    socktype = socket.SOCK_STREAM
    sock = socket.socket(sockfamily, socktype)
    sock.settimeout(1.0)
    try:
        sock.connect((host, port))
        return True
    except socket.error:
        return False
    finally:
        sock.close()


class PingMiddleware(object):
    PING_RE = re.compile(r'^/ping/?$')
    PING_DB_RE = re.compile(r'^/ping_db/?$')

    def __init__(self):
        self.db_hosts_to_check = [
            tuple(line.split(':'))
            for line in settings.MONGO_HOSTS.split(',')
        ]

    def process_request(self, request):
        if self.PING_RE.match(request.path):
            return HttpResponse('OK')

        if self.PING_DB_RE.match(request.path):
            for host, port in self.db_hosts_to_check:
                if check_host(host, int(port)):
                    return HttpResponse('OK')

            return HttpResponseServerError('NOTOK')


try:
    import uwsgi
except ImportError:
    uwsgi = None


class UwsgiLogMiddleware(object):
    USER_IP_HEADERS = (
        'HTTP_X_REAL_IP',
        'HTTP_X_FORWARDED_FOR',
        'REMOTE_ADDR',
    )

    HOST_HEADERS = (
        'HTTP_X_FORWARDED_HOST',
        'HTTP_HOST',
    )

    @staticmethod
    def extract_actual_header(request, headers):
        for header_name in headers:
            ip = request.META.get(header_name)
            if ip:
                return ip
        return None

    if uwsgi:

        def process_request(self, request):
            try:
                yandex_login = request.COOKIES.get('yandex_login')
                if yandex_login:
                    uwsgi.set_logvar('yandex_login', yandex_login)

                user_ip = self.extract_actual_header(request, self.USER_IP_HEADERS)
                if user_ip:
                    uwsgi.set_logvar('user_ip', user_ip)

                host = self.extract_actual_header(request, self.HOST_HEADERS)
                if host:
                    uwsgi.set_logvar('real_host', host)
            except Exception:
                logger.exception('Exception on set uWSGI logvar')

        def process_view(self, request, *args, **kwargs):
            try:
                uwsgi.set_logvar('view_name', request.resolver_match.view_name)

                request_data = _get_request_data(request)
                logger.info('%s?%s', request.path, urlencode(request_data, 'utf-8'))

                client_application = getattr(request, 'client_application', None)
                if client_application and getattr(client_application, 'id', None):
                    uwsgi.set_logvar('client_application', client_application.name.encode('utf-8'))
                    uwsgi.set_logvar('client_id', client_application.id)
                user = getattr(request, 'yauser', None)
                if user:
                    if isinstance(user, TvmServiceRequest):
                        uwsgi.set_logvar('tvm_client_id', str(user.service_ticket.src))
                    else:
                        uwsgi.set_logvar('yandex_login', user.get_username())

            except Exception:
                logger.exception('Exception on set uWSGI logvar')


class AuthMiddleware(YandexAuthBackendMiddleware):
    def process_request(self, request):
        super(AuthMiddleware, self).process_request(request)

        if request.yauser is None or not request.yauser.is_authenticated():
            return HttpResponse(status=401)

        if not isinstance(request.yauser, TvmServiceRequest):
            if request.yauser.get_username() in settings.BAN_LIST:
                return HttpResponse(status=418)


class TestAuthMiddleware(object):
    """
    Аутентификация для использования в тестах
    """
    def process_request(self, request):
        email = 'awesome@yandex-team.ru'
        mechanism_name = 'cookie'
        service_ticket = None

        class FakeMechanism(object):
            def __init__(self):
                self.mechanism_name = mechanism_name

        request.yauser = YandexUser(
            uid='000000',
            is_lite=False,
            fields={},
            need_reset=False,
            emails=[email],
            default_email=email,
            mechanism=FakeMechanism(),
            service_ticket=service_ticket,
        )
