# coding: utf-8
from __future__ import unicode_literals

import logging
from static_api.dehydrators import Dehydrator

from .person import PersonRefDehydratorMixin, PersonNewRefDehydrator
from ..models import group


log = logging.getLogger(__name__)


class DepartmentRefDehydrator(Dehydrator):
    exportable_roles = set(group.DepartmentStaffModel.role_mapping.values())

    fields = [
        'is_deleted',
        'url',

        'level',

        'description',

        'id',
        'name',
        'contacts',
        'heads',
        'kind',
    ]

    def get_heads(self):
        return [
            DepartmentStaffRefDehydrator(head).as_dict()
            for head in self.data.get('heads', [])
            if head['role'] in self.exportable_roles
        ]

    def get_kind(self):
        if not self.data.get('kind'):
            return None

        return DepartmentKindRefDehydrator(self.data['kind']).as_dict()


class DepartmentDehydrator(DepartmentRefDehydrator):
    pass


class DepartmentShortDehydrator(Dehydrator):
    fields = [
        'id',
    ]


class DepartmentKindRefDehydrator(Dehydrator):
    fields = [
        'id',
        'slug',
        'name'
    ]


class DepartmentKindDehydrator(DepartmentKindRefDehydrator):
    fields = DepartmentKindRefDehydrator.fields + ['rank']


class DepartmentChainDehydrator(Dehydrator):
    fields = [
        'id',
        'department',
        'chiefs',
        'hr_partners',
    ]

    def get_chiefs(self):
        return [PersonNewRefDehydrator(chief).as_dict() for chief in self.data['chiefs']]

    def get_hr_partners(self):
        return [PersonNewRefDehydrator(hr_partner).as_dict() for hr_partner in self.data['hr_partners']]


class AncestorsMixin(object):
    def get_ancestors(self):
        return [GroupRefNestedDehydrator(group).as_dict() for group in self.data.get('ancestors', [])]


class GroupShortRefNestedDehydrator(Dehydrator):
    fields = (
        'is_deleted',
        'name',
        'url',
        'level',

        'id',
        'type',
        'service',
        'department',
        'role_scope',
    )
    _department_dehydrator = DepartmentShortDehydrator

    def get_department(self):
        if not self.data.get('department'):
            return {'id': None}

        if not self.data['department'].get('id'):
            return {'id': None}

        return self._department_dehydrator(self.data['department']).as_dict()


class GroupRefNestedDehydrator(GroupShortRefNestedDehydrator):
    fields = GroupShortRefNestedDehydrator.fields + (
        'description',
        'affiliation_counters',
        'responsibles',
    )
    _department_dehydrator = DepartmentRefDehydrator

    def get_responsibles(self):
        return [GroupResponsibleRefDehydrator(resp).as_dict() for resp in self.data.get('responsibles', [])]


class GroupRefDehydrator(AncestorsMixin, GroupRefNestedDehydrator):
    fields = GroupRefNestedDehydrator.fields + ('parent', 'ancestors',)

    def get_parent(self):
        parent = self.data.get('parent')

        if parent:
            return GroupRefNestedDehydrator(parent).as_dict()


class GroupDehydrator(GroupRefDehydrator):
    fields = GroupRefDehydrator.fields + ('_meta',)


class GroupAncestorsMixin(object):
    def get_group(self):
        return GroupRefDehydrator(self.data['group']).as_dict()


class GroupMembershipDehydrator(GroupAncestorsMixin, PersonRefDehydratorMixin, Dehydrator):
    fields = (
        '_meta',
        'joined_at',
        'id',
        'group',
        'person',
    )


class GroupResponsibleRefDehydrator(PersonRefDehydratorMixin, Dehydrator):
    fields = (
        'permitted_at',
        'id',
        'person',
    )


class GroupResponsibleDehydrator(GroupResponsibleRefDehydrator):
    fields = GroupResponsibleRefDehydrator.fields + ('group',)


class GroupMembershipPersonDehydrator(GroupMembershipDehydrator):
    fields = (
        'joined_at',
        'id',
        'person',
    )


class GroupMembershipGroupDehydrator(GroupMembershipDehydrator):
    fields = (
        'joined_at',
        'id',
        'group',
    )


class DepartmentStaffRefDehydrator(PersonRefDehydratorMixin, Dehydrator):
    fields = (
        'id',
        'role',
        'person',
    )


class DepartmentStaffDehydrator(DepartmentStaffRefDehydrator):
    fields = DepartmentStaffRefDehydrator.fields + ('department_group',)

    def get_department_group(self):
        if not self.data.get('department_group'):
            log.error("department_group is None")
            return {'id': None}

        return GroupShortRefNestedDehydrator(self.data['department_group']).as_dict()
