# coding: utf-8
import pymongo
from bson.objectid import ObjectId
from urlparse import urlparse, parse_qs

from django.conf import settings
from staff_api.v3_0.idm.db_collections import idm_roles_collection
from static_api.request_parsers import SpecialParamsParser
from static_api.request_parsers.special_params import _query_fields_gen
from static_api import (
    storage,
    resources,
    utils as static_utils,
)


def validate_idm_role_record(role_record):
    expected_fields = ('role_type', 'subject', 'subject_type', 'resource', 'role', 'fields', 'deleted', 'handled')
    return all(field in role_record for field in expected_fields)


def extract_fields_for_resource_from_url(resource_name, access_url):
    resource = resources.registry.by_name[resource_name]
    parsed_url = urlparse(access_url)
    url_params = parse_qs(parsed_url.query)
    params = {param: ','.join(values) for param, values in url_params.items()}
    params_parser = SpecialParamsParser(resource, {settings.STATIC_API_WILDCARD_FIELD_ACCESS}, params)
    if not params_parser.is_valid():
        return None
    cleaned = params_parser.cleaned_data_obj
    query_fields = set(_query_fields_gen(cleaned.query))
    sort_fields = {field.strip('-+') for field in cleaned.sort}
    request_fields = set(params.get('_fields', '').split(','))
    return set.union(request_fields, query_fields, sort_fields)


@static_utils.cached_function(cache_time=60 * 60 * 24 * 7)
def get_person_uid_by_login(login):
    namespace = storage.manager.get_read_namespace()
    person_record = (
        namespace['person']
        .get_one(
            lookup={'login': login},
            fields={'_id': 0, 'uid': 1},
        )
    )
    return person_record and person_record['uid']


@static_utils.cached_function(cache_time=60 * 60 * 24 * 7)
def get_person_login_by_id(person_id):
    namespace = storage.manager.get_read_namespace()
    person_record = (
        namespace['person']
        .get_one(
            lookup={'id': person_id},
            fields={'_id': 0, 'login': 1},
        )
    )
    return person_record and person_record['login']


@static_utils.cached_function(cache_time=60 * 60 * 24 * 7)
def get_person_id_by_login(login):
    namespace = storage.manager.get_read_namespace()
    person_record = (
        namespace['person']
        .get_one(
            lookup={'login': login},
            fields={'_id': 0, 'id': 1},
        )
    )
    return person_record and person_record['id']


def combine_dicts(*dicts):
    new_dict = dicts[0].copy()
    for d in dicts[1:]:
        new_dict.update(d)
    return new_dict


def get_roles_chunk(from_id='', limit=10000):
    filter_conditions = {'deleted': False}
    if from_id:
        filter_conditions['_id'] = {'$gt': ObjectId(from_id)}
    projection = {
        '_id': 1,
        'role_type': 1,
        'resource': 1,
        'role': 1,
        'subject_type': 1,
        'subject': 1,
        'fields': 1,
    }
    raw_roles = (
        idm_roles_collection
        .find(filter=filter_conditions, projection=projection)
        .sort([('_id', pymongo.ASCENDING)])
        .limit(limit)
    )
    return raw_roles
