# coding: utf-8
from copy import deepcopy
from django.core.urlresolvers import reverse

from django_idm_api.exceptions import UnsupportedApi, BadRequest
from django_idm_api.hooks import BaseHooks

from staff_api.v3_0.idm.base import BaseActions
from staff_api.v3_0.idm.registry import ActionRegistry
from staff_api.v3_0.idm.resource_access_actions import (
    ResourceAccessActions,
    ResourceFullAccessActions,
    ResourcePartialAccessActions,
)

from staff_api.v3_0.idm.helpers import get_person_id_by_login, get_roles_chunk


ROLES_TREE = {
    'code': 0,
    'roles': {
        'slug': 'role_type',
        'name': {'en': 'Role type', 'ru': 'Тип роли'},
        'values': {
            'resource_access': {
                '_actions_cls': ResourceAccessActions,
                'help': {'en': 'ACL for data', 'ru': 'Доступ к данным'},
                'name': {'en': 'Access', 'ru': 'Доступ'},
                'roles': {
                    'slug': 'resource',
                    'name': {'en': 'Resource', 'ru': 'Сущность'},
                    'values': {
                        '__id__': {
                            'name': {'en': '__name__', 'ru': '__name__'},
                            'roles': {
                                'slug': 'access_type',
                                'name': {'en': 'Access type', 'ru': 'Тип доступа'},
                                'values': {
                                    'full_access': {
                                        '_actions_cls': ResourceFullAccessActions,
                                        'help': {
                                            'en': 'Access to all fields',
                                            'ru': 'Полный доступ к коллекции'
                                        },
                                        'name': {'en': 'Full access', 'ru': 'Полный доступ'},
                                        'set': 'full_access',
                                    },
                                    'partial_access': {
                                        '_actions_cls': ResourcePartialAccessActions,
                                        'help': {
                                            'en': 'Request acces to particular fields',
                                            'ru': 'Запрос доступа к отдельным полям'
                                        },
                                        'name': {'en': 'Access to fields', 'ru': 'Доступ к полям'},
                                        'set': 'partial_access',
                                        'fields': [
                                            {
                                                'slug': 'access_url',
                                                'name': {
                                                    'ru': 'Ссылка в staff-api с перечислением полей `_fields`',
                                                    'en': 'staff-api url with `_fields` param',
                                                },
                                                'type': 'charfield',
                                                'required': True,
                                                'options': {'blank_allowed': False},
                                            },
                                        ],
                                    },
                                },
                            },
                        },
                    },
                },
            },
        },
    },
}


class Hooks(BaseHooks):
    GET_ROLES_PAGE_SIZE = 10000

    def info(self):
        """Вернуть полное дерево ролей"""
        actions = self._get_actions()
        info = actions.get_info()
        return info

    def add_role_impl(self, login, role, fields, **kwargs):
        """Добавить человеку роль"""
        person_id = get_person_id_by_login(login)
        if not person_id:
            raise BadRequest('Unknown user: {}'.format(login))

        actions = self._get_actions()
        actions.add_role(
            subject_id=person_id,
            subject_type='user',
            role=role,
            fields=fields,
        )
        return {'data': fields}

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        """Удалить роль человека"""
        person_id = get_person_id_by_login(login)
        actions = self._get_actions()
        actions.remove_role(
            subject_id=person_id,
            subject_type='user',
            role=role,
            fields=data,
        )

    def get_all_roles_impl(self, **kwargs):
        """Перечислить всех людей и их роли"""
        result = {}
        actions = self._get_actions()
        for subject_type, subject_id, role, fields in actions.get_all_roles():
            if subject_id not in result:
                result[subject_id] = {'subject_type': subject_type, 'roles': []}
            if fields:
                result[subject_id]['roles'].append([role, fields])
            else:
                result[subject_id]['roles'].append(role)
        return list(result.items())

    def add_tvm_role_impl(self, login, role, fields, **kwargs):
        """добавить роль tvm-приложению с tvm_id == login"""
        actions = self._get_actions()
        actions.add_role(
            subject_id=int(login),
            subject_type='tvm_app',
            role=role,
            fields=fields,
        )

    def remove_tvm_role_impl(self, login, role, data, is_fired, **kwargs):
        """добавить роль tvm-приложению с tvm_id == login"""
        actions = self._get_actions()
        actions.remove_role(
            subject_id=int(login),
            subject_type='tvm_app',
            role=role,
            fields=data,
        )

    def get_all_roles(self):
        """Переопределяем для поддержки subject_type"""
        return {
            'code': 0,
            'users': [
                {
                    'login': subject_id,
                    'subject_type': data['subject_type'],
                    'roles': data['roles'],
                } for subject_id, data in self.get_all_roles_impl()
            ]
        }

    def get_roles(self, request, *args, **kwargs):

        from_id = request.GET.get('from_id')
        limit = int(request.GET.get('limit', self.GET_ROLES_PAGE_SIZE))

        roles_chunk = list(get_roles_chunk(from_id, limit=limit))

        result = {
            'code': 0,
            'roles': self.get_roles_impl(roles_chunk),
        }

        if len(roles_chunk) == limit:
            result['next-url'] = self.get_next_url(from_id=roles_chunk[-1]['_id'])
        return result

    def get_roles_impl(self, roles, **kwargs):
        result = []
        for role in roles:
            action = ActionRegistry.get(role['role_type'], role['role'])
            role_dict = {
                'path': '/{role_type}/{resource}/{role}/'.format(**role),
                'login': role['subject'],
            }
            if action.role_fields:
                role_dict['fields'] = {
                    field_name: role['fields'].get(field_name, '')
                    for field_name in action.role_fields
                }
            result.append(role_dict)
        return result

    @staticmethod
    def get_next_url(from_id):
        return '{uri}?from_id={from_id}'.format(
            uri=reverse('idm-api:get-roles'),
            from_id=str(from_id),
        )

    @staticmethod
    def _get_actions():
        roles_tree = deepcopy(ROLES_TREE)
        actions = BaseActions(tree_node=roles_tree, key_crumbs=[])
        return actions
