# coding: utf-8
import pymongo
import logging

from django_idm_api.exceptions import RoleNotFound

from static_api.resources import registry as resource_registry

from staff_api.v3_0.idm.base import BaseActions
from staff_api.v3_0.idm.db_collections import idm_roles_collection
from staff_api.v3_0.idm.helpers import get_person_login_by_id, combine_dicts
from staff_api.v3_0.idm.registry import ActionRegistry


logger = logging.getLogger(__name__)
role_type_filter = {'role_type': 'resource_access', 'deleted': False}


class ResourceAccessActions(BaseActions):
    @staticmethod
    def get_objects(fields):
        resources = resource_registry.by_name
        return [
            {
                'id': key.encode(),
                'name': resource.verbose.encode()
            }
            for key, resource in resources.items()
        ]


class ResourceAccessBaseActions(BaseActions):
    role_filter = {'role': 'full_access'}
    role_fields = []

    @staticmethod
    def get_objects(fields):
        objects = idm_roles_collection.find(role_type_filter, projection={f: 1 for f in fields})
        return list(objects)

    @staticmethod
    def _check_resource(resource_name):
        if resource_name not in resource_registry.by_name:
            raise RoleNotFound('Role "resource:%s" not found' % resource_name)

    def _get_role_kwargs(self, subject_id, subject_type, role, fields=None):
        """Вернуть dict в том виде, в котором он в монге лежит"""
        fields = fields or {}
        is_user = subject_type == 'user'

        resource = role['resource']
        self._check_resource(role['resource'])

        role_kwargs = {
            'resource': resource,
            'role': role['access_type'],
            'subject': get_person_login_by_id(subject_id) if is_user else subject_id,
            'fields': {},
            'subject_type': subject_type,
        }
        role_kwargs.update(self.role_filter)

        for field_name in self.role_fields:
            role_kwargs['fields'][field_name] = fields[field_name]

        return role_kwargs

    def _get_or_create_role(self, **role_kwargs):
        created = False
        role_dict = combine_dicts(role_kwargs, {'handled': False, 'deleted': False})
        existing = idm_roles_collection.find_one(role_kwargs)
        if not existing:
            idm_roles_collection.insert_one(role_dict)
            created = True

        return created

    def _remove_role(self, **role_kwargs):
        existing = idm_roles_collection.find_one(role_kwargs)
        if not existing:
            logger.warning('Cannot find role to remove. Role:%s', role_kwargs)
            return
        idm_roles_collection.update_one(
            filter={'_id': existing['_id']},
            update={'$set': {'handled': False, 'deleted': True}}
        )

    def add_role(self, subject_id, subject_type, role, fields=None):
        role_kwargs = role_type_filter.copy()
        role_kwargs.update(self._get_role_kwargs(subject_id, subject_type, role, fields))
        created = self._get_or_create_role(**role_kwargs)
        return created

    def remove_role(self, subject_id, subject_type, role, fields=None):
        role_kwargs = role_type_filter.copy()
        role_kwargs.update(self._get_role_kwargs(subject_id, subject_type, role, fields))
        self._remove_role(**role_kwargs)

    def get_all_roles(self):
        roles_filter = role_type_filter.copy()
        roles_filter.update(self.role_filter)

        projection = {
            '_id': 0,
            'subject_type': 1,
            'subject': 1,
            'resource': 1,
            'role': 1,
            'fields': 1,
        }
        for field_name in self.role_fields:
            projection[field_name] = 1

        all_roles = (
            idm_roles_collection
            .find(
                filter=roles_filter,
                projection=projection,
            )
            .sort([
                ('subject', pymongo.ASCENDING),
                ('resource', pymongo.ASCENDING),
            ])
        )
        for role_record in all_roles:
            key_crumbs = dict(self.key_crumbs)
            key_crumbs['resource'] = str(role_record['resource'])
            fields_data = role_record.get('fields', {})
            fields_data = {f: fields_data.get(f, 'STUB') for f in self.role_fields}
            yield role_record['subject_type'], role_record['subject'], key_crumbs, fields_data


@ActionRegistry.register(role_type='resource_access', role='full_access')
class ResourceFullAccessActions(ResourceAccessBaseActions):
    role_filter = {'role': 'full_access'}
    role_fields = []


@ActionRegistry.register(role_type='resource_access', role='partial_access')
class ResourcePartialAccessActions(ResourceAccessBaseActions):
    role_filter = {'role': 'partial_access'}
    role_fields = ['access_url', ]
