# coding: utf-8
import logging
from collections import namedtuple
from django.conf import settings

from pymongo.errors import BulkWriteError

from staff_api.v3_0.idm.db_collections import idm_roles_collection, access_rules_collection
from staff_api.v3_0.idm.helpers import (
    get_person_uid_by_login,
    extract_fields_for_resource_from_url,
    validate_idm_role_record,
)


logger = logging.getLogger(__name__)


IDM_ROLE_FIELDS_BASE = [
    'role',
    'subject',
    'subject_type',
    'role_type',
    'resource',
    'fields',
    'deleted',
    'handled',
]

IDMRole = namedtuple(
    typename='IDMRole',
    field_names=IDM_ROLE_FIELDS_BASE + ['id'],
)

FieldsAccessSyncResult = namedtuple(
    typename='FieldsAccessSyncResult',
    field_names=['access_seen', 'access_delete', 'access_insert', 'idm_delete', 'idm_process', 'idm_seen'],
)


class FieldsAccessSync(object):

    access_rules_collection = access_rules_collection
    idm_roles_collection = idm_roles_collection

    def __init__(self, idm_roles):
        self.idm_roles = idm_roles

        self.result = FieldsAccessSyncResult([], [], [], [], [], [])

    def _get_subject_id(self, role_obj):
        if role_obj.subject_type == 'tvm_app':
            return role_obj.subject
        elif role_obj.subject_type == 'user':
            return get_person_uid_by_login(role_obj.subject)
        else:
            raise ValueError('Invalid subject type: %s' % role_obj.subject_type)

    def _get_idm_role_fields(self, role_obj):
        if role_obj.role == 'full_access':
            fields = [settings.STATIC_API_WILDCARD_FIELD_ACCESS]
        else:
            fields = extract_fields_for_resource_from_url(
                resource_name=role_obj.resource,
                access_url=role_obj.fields.get('access_url', ''),
            )
        return fields

    def _process_role_fields(self, role_obj, subject_id, fields):
        if fields is None:
            logger.warning('Skipping malformed role record: %s' % role_obj)
            return

        for field in fields:
            access_rule = {
                'subject_id': str(subject_id),
                'resource': role_obj.resource,
                'subject_type': role_obj.subject_type,
                'field': field,
                'idm_role_id': role_obj.id,
            }
            self.result.access_insert.append(access_rule)

    def process_roles(self):
        for role_record in self.idm_roles:
            if not validate_idm_role_record(role_record):
                logger.warning('Skipping malformed role record: %s' % role_record)
                continue

            role_record['id'] = role_record.pop('_id')
            role_obj = IDMRole(**role_record)
            self.result.idm_seen.append(role_obj.id)
            if role_obj.deleted:
                self.result.access_delete.append(role_obj.id)
            else:
                subject_id = self._get_subject_id(role_obj)
                fields = self._get_idm_role_fields(role_obj)
                self._process_role_fields(role_obj, subject_id, fields)
            self.result.idm_process.append(role_obj.id)

    def finalize(self):
        if self.result.access_insert:
            try:
                self.access_rules_collection.insert_many(self.result.access_insert, ordered=False)
            except BulkWriteError:
                pass

        if self.result.access_delete:
            self.access_rules_collection.delete_many({'idm_role_id': {'$in': self.result.access_delete}})

        if self.result.idm_process:
            self.idm_roles_collection.update_many(
                filter={'_id': {'$in': self.result.idm_process}},
                update={'$set': {'handled': True}},
            )

        self.idm_roles_collection.delete_many({'deleted': True, 'handled': True})
        self.result = FieldsAccessSyncResult([], [], [], [], [], [])

    def sync(self):
        self.process_roles()
        self.finalize()
