# coding: utf-8
from __future__ import unicode_literals

import logging

from django.conf import settings

from static_api.listeners import register, BaseListener

from .. import models, dehydrators

log = logging.getLogger(__name__)


@register('django_intranet_stuff.groupmembership')
class GroupMembershipListener(BaseListener):
    model_class = models.group.GroupMembershipModel

    def on_modify(self, model, ns):
        if model.group['id'] in settings.API_IGNORE_GROUPS:
            return

        ns.groupmembership.put({'id': model.id},
                               data=model.as_dict(dehydrators.group.GroupMembershipDehydrator))

        ns.person.put_nested('groups.$',
                             lookup={'groups.$.id': model.id},
                             parent_lookup={'id': model.person['id']},
                             data=model.as_dict(dehydrators.group.GroupMembershipGroupDehydrator))

    def on_delete(self, model, ns):
        ns.groupmembership.delete({'id': model.id})

        ns.person.delete_nested(
            'groups.$',
            lookup={'groups.$.id': model.id}
        )


@register('django_intranet_stuff.departmentstaff')
class DepartmentStaffListener(BaseListener):
    model_class = models.group.DepartmentStaffModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):

        ns.departmentstaff.put({'id': model.id},
                               data=model.as_dict(dehydrators.group.DepartmentStaffDehydrator))

        if model.get_raw('role') not in model.role_mapping:
            return

        data_ref = model.as_dict(dehydrators.group.DepartmentStaffRefDehydrator)

        def put_ref(collection, path, parent_path):
            collection.put_nested(
                path,
                lookup={path + '.id': model.id},
                parent_lookup={parent_path: model.department},
                data=data_ref,
            )

        put_ref(ns.department, 'heads.$', 'id')

        put_ref(ns.group, 'department.heads.$', 'department.id')
        put_ref(ns.group, 'parent.department.heads.$', 'parent.department.id')
        put_ref(ns.group, 'ancestors.$.department.heads.$', 'ancestors.$.department.id')

        put_ref(ns.person, 'department_group.department.heads.$', 'department_group.department.id')
        put_ref(ns.person, 'department_group.parent.department.heads.$', 'department_group.parent.department.id')
        put_ref(ns.person, 'department_group.ancestors.$.department.heads.$',
                'department_group.ancestors.$.department.id')
        put_ref(ns.person, 'groups.$.group.department.heads.$', 'groups.$.group.department.id')
        put_ref(ns.person, 'groups.$.group.parent.department.heads.$', 'groups.$.group.parent.department.id')
        put_ref(ns.person, 'groups.$.group.ancestors.$.department.heads.$', 'groups.$.group.ancestors.$.department.id')

        put_ref(ns.groupmembership, 'group.department.heads.$', 'group.department.id')
        put_ref(ns.groupmembership, 'group.parent.department.heads.$', 'group.parent.department.id')
        put_ref(ns.groupmembership, 'group.ancestors.$.department.heads.$', 'group.ancestors.$.department.id')

    def on_delete(self, model, ns):
        ns.departmentstaff.delete({'id': model.id})

        if model.get_raw('role') not in model.role_mapping:
            return

        def delete_ref(collection, path):
            return collection.update_nested_array_element(path, model.id, delete=True)

        ns.department.execute_bulk([delete_ref(ns.department, 'heads.$')], ordered=False)

        ops = [
            delete_ref(ns.group, 'department.heads.$'),
            delete_ref(ns.group, 'parent.department.heads.$'),
            delete_ref(ns.group, 'ancestors.$.department.heads.$'),
        ]
        ns.group.execute_bulk(ops, ordered=False)

        ops = [
            delete_ref(ns.person, 'department_group.department.heads.$'),
            delete_ref(ns.person, 'department_group.parent.department.heads.$'),
            delete_ref(ns.person, 'department_group.ancestors.$.department.heads.$'),
            delete_ref(ns.person, 'groups.$.department.heads.$'),
            delete_ref(ns.person, 'groups.$.parent.department.heads.$'),
            delete_ref(ns.person, 'groups.$.ancestors.$.department.heads.$'),
        ]
        ns.person.execute_bulk(ops, ordered=False)

        ops = [
            delete_ref(ns.groupmembership, 'group.department.heads.$'),
            delete_ref(ns.groupmembership, 'group.parent.department.heads.$'),
            delete_ref(ns.groupmembership, 'group.ancestors.$.department.heads.$'),
        ]
        ns.groupmembership.execute_bulk(ops, ordered=False)


@register('django_intranet_stuff.groupresponsible')
class GroupResponsibleListener(BaseListener):
    model_class = models.group.GroupResponsibleModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.groupresponsible.put({'id': model.id},
                                data=model.as_dict(dehydrators.group.GroupResponsibleDehydrator))

        def put_ref(collection, path, parent_path):
            collection.put_nested(path, lookup={path + '.id': model.id},
                                  parent_lookup={parent_path: model.group},
                                  data=model.as_dict(dehydrators.group.GroupResponsibleRefDehydrator))

        put_ref(ns.group, 'responsibles.$', 'id')
        put_ref(ns.group, 'parent.responsibles.$', 'parent.id')
        put_ref(ns.group, 'ancestors.$.responsibles.$', 'ancestors.$.id')

        put_ref(ns.person, 'groups.$.group.responsibles.$', 'groups.$.group.id')
        put_ref(ns.person, 'groups.$.group.parent.responsibles.$', 'groups.$.group.parent.id')
        put_ref(ns.person, 'groups.$.group.ancestors.$.responsibles.$', 'groups.$.group.ancestors.$.id')

        put_ref(ns.groupmembership, 'group.responsibles.$', 'group.id')
        put_ref(ns.groupmembership, 'group.ancestors.$.responsibles.$', 'group.ancestors.$.id')

    def on_delete(self, model, ns):
        ns.groupresponsible.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, lookup={path + '.id': model.id})

        delete_ref(ns.group, 'responsibles.$')
        delete_ref(ns.group, 'parent.responsibles.$')
        delete_ref(ns.group, 'ancestors.$.responsibles.$')

        delete_ref(ns.person, 'groups.$.group.responsibles.$')
        delete_ref(ns.person, 'groups.$.group.parent.responsibles.$')
        delete_ref(ns.person, 'groups.$.group.ancestors.$.responsibles.$')

        delete_ref(ns.groupmembership, 'group.responsibles.$')
        delete_ref(ns.groupmembership, 'group.parent.responsibles.$')
        delete_ref(ns.groupmembership, 'group.ancestors.$.responsibles.$')


@register('django_intranet_stuff.group')
class GroupListener(BaseListener):
    model_class = models.group.GroupModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        if model.level == 0:
            log.info('Ignored group %s: level == %s', model.id, model.level)
            return

        if model.id in settings.API_IGNORE_GROUPS:
            log.info('Ignored group %s: id in API_IGNORE_GROUPS', model.id)
            return

        group_data = model.as_dict(dehydrators.group.GroupDehydrator)
        ns.group.put({'id': model.id}, data=group_data)

        group_ref_data = model.as_dict(dehydrators.group.GroupRefDehydrator)
        group_ref_nested_data = model.as_dict(dehydrators.group.GroupRefNestedDehydrator)

        def update_in_nested_array_element(collection, path, data):
            return collection.update_in_nested_array_element(path, model.id, data=data)

        def update_nested_array_element(collection, path, data):
            return collection.update_nested_array_element(path, model.id, data=data)

        def update_nested(collection, path, data):
            return collection.update_nested(path, model.id, data=data)

        ops = [
            update_nested(ns.group, 'parent', group_ref_nested_data),
            update_nested_array_element(ns.group, 'ancestors.$', group_ref_nested_data),
        ]
        ns.group.execute_bulk(ops, ordered=False)

        ops = [
            update_in_nested_array_element(ns.person, 'groups.$.group', group_ref_data),
            update_in_nested_array_element(ns.person, 'groups.$.group.parent', group_ref_nested_data),
            update_nested_array_element(ns.person, 'groups.$.group.ancestors.$', group_ref_nested_data),
            update_nested_array_element(ns.person, 'department_group.ancestors.$', group_ref_nested_data),
        ]

        if model.department['id']:
            ns.person.put_nested(
                'department_group',
                 lookup={'department_group.department.id': model.department['id']},
                 data=model.as_dict(dehydrators.group.GroupRefDehydrator),
            )

        ops.append(update_nested(ns.person, 'department_group.parent', group_ref_data))
        ns.person.execute_bulk(ops, ordered=False)

        ops = [
            update_nested(ns.groupmembership, 'group', group_ref_data),
            update_nested(ns.groupmembership, 'group.parent', group_ref_nested_data),
            update_nested_array_element(ns.groupmembership, 'group.ancestors.$', group_ref_nested_data),
        ]
        ns.groupmembership.execute_bulk(ops, ordered=False)

    def on_delete(self, model, ns):
        ns.group.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, lookup={path + '.id': model.id})

        delete_ref(ns.group, 'parent')
        delete_ref(ns.group, 'ancestors.$')
        delete_ref(ns.group, 'ancestors.$.parent')

        delete_ref(ns.person, 'groups.$.group')
        delete_ref(ns.person, 'groups.$.group.parent')
        delete_ref(ns.person, 'groups.$.group.ancestors.$')

        if model.department['id']:
            ns.person.delete_nested('department_group',
                                    lookup={'department_group.department.id': model.department['id']})

        delete_ref(ns.person, 'department_group.parent')
        delete_ref(ns.person, 'department_group.ancestors.$')

        delete_ref(ns.groupmembership, 'group')
        delete_ref(ns.groupmembership, 'group.parent')
        delete_ref(ns.groupmembership, 'group.ancestors.$')


@register('django_intranet_stuff.department')
class DepartmentListener(BaseListener):
    model_class = models.group.DepartmentModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        instance_class = model.get_raw('instance_class', 'department')

        if instance_class == 'department':
            self.modify_department(model, ns)
        elif instance_class == 'geography':
            self.modify_geography(model, ns)

    def on_delete(self, model, ns):
        instance_class = model.get_raw('instance_class', 'department')

        if instance_class == 'department':
            self.delete_department(model, ns)
        elif instance_class == 'geography':
            self.delete_geography(model, ns)

    def modify_department(self, model, ns):
        ns.department.put({'id': model.id}, data=model.as_dict(dehydrators.group.DepartmentDehydrator))
        ref_data = model.as_dict(dehydrators.group.DepartmentRefDehydrator)

        ops = [
            self.update_nested_ref(ns.group, 'department', model, ref_data),
            self.update_nested_ref(ns.group, 'parent.department', model, ref_data),
            self.update_in_nested_elements_ref(ns.group, 'ancestors.$.department', model, ref_data)
        ]
        ns.group.execute_bulk(ops, ordered=False)

        ops = [
            self.update_nested_ref(ns.person, 'department_group.department', model, ref_data),
            self.update_nested_ref(ns.person, 'department_group.parent.department', model, ref_data),
            self.update_in_nested_elements_ref(ns.person, 'department_group.ancestors.$.department', model, ref_data),

            self.update_in_nested_elements_ref(ns.person, 'groups.$.group.department', model, ref_data),
            self.update_in_nested_elements_ref(ns.person, 'groups.$.group.parent.department', model, ref_data),
            self.update_in_nested_elements_ref(ns.person, 'groups.$.group.ancestors.$.department', model, ref_data),
        ]
        ns.person.execute_bulk(ops, ordered=False)

        ops = [
            self.update_nested_ref(ns.groupmembership, 'group.department', model, ref_data),
            self.update_nested_ref(ns.groupmembership, 'group.parent.department', model, ref_data),
            self.update_in_nested_elements_ref(ns.groupmembership, 'group.ancestors.$.department', model, ref_data),
        ]
        ns.groupmembership.execute_bulk(ops, ordered=False)

        self.put_ref(ns.departmentstaff, 'department_group.department', model)

    def modify_geography(self, model, ns):
        ref_data = model.as_dict(lambda m: dehydrators.geography.GeographyDehydrator(m, ns))
        ns.geography.put({'id': model.id}, data=ref_data)

        ops = [
            self.update_nested_ref(ns.geography, 'parent', model, ref_data),
            self.update_nested_array_element(ns.geography, 'ancestors.$', model, ref_data)
        ]
        ns.geography.execute_bulk(ops, ordered=False)

    def delete_department(self, model, ns):
        ns.department.delete({'id': model.id})

        self.delete_ref(ns.group, 'department', model)
        self.delete_ref(ns.group, 'parent.department', model)
        self.delete_ref(ns.group, 'ancestors.$.department', model)

        self.delete_ref(ns.person, 'department_group.department', model)
        self.delete_ref(ns.person, 'department_group.parent.department', model)
        self.delete_ref(ns.person, 'department_group.ancestors.$.department', model)
        self.delete_ref(ns.person, 'groups.$.group.department', model)
        self.delete_ref(ns.person, 'groups.$.group.parent.department', model)
        self.delete_ref(ns.person, 'groups.$.group.ancestors.$.department', model)

        self.delete_ref(ns.groupmembership, 'group.department', model)
        self.delete_ref(ns.groupmembership, 'group.parent.department', model)
        self.delete_ref(ns.groupmembership, 'group.ancestors.$.department', model)

        self.delete_ref(ns.departmentstaff, 'department_group.department', model)

    def delete_geography(self, model, ns):
        ns.geography.delete({'id': model.id})

        self.delete_ref(ns.geography, 'parent', model)
        self.delete_ref(ns.geography, 'ancestors.$', model)
        self.delete_ref(ns.geography, 'ancestors.$.parent', model)

    @staticmethod
    def delete_ref(collection, path, model):
        collection.delete_nested(path, lookup={path + '.id': model.id})

    @staticmethod
    def put_ref(collection, path, model):
        collection.put_nested(
            path,
            lookup={path + '.id': model.id},
            data=model.as_dict(dehydrators.group.DepartmentShortDehydrator)
        )

    @staticmethod
    def update_nested_ref(collection, path, model, ref_data):
        return collection.update_nested(path, model.id, ref_data)

    @staticmethod
    def update_in_nested_elements_ref(collection, path, model, ref_data):
        return collection.update_in_nested_array_element(path, model.id, ref_data)

    @staticmethod
    def update_nested_array_element(collection, path, model, data):
        return collection.update_nested_array_element(path, model.id, data=data)


@register('django_intranet_stuff.departmentkind')
class DepartmentKindListener(BaseListener):
    model_class = models.group.DepartmentKindModel
    delete_on_zero_intranet_status = False

    def on_modify(self, model, ns):
        ns.departmentkind.put({'id': model.id},
                              data=model.as_dict(dehydrators.group.DepartmentKindDehydrator))

        if model.id not in settings.API_DEPARTMET_KINDS:
            return

        def put_ref(collection, path):
            collection.put_nested(path, lookup={path + '.id': model.id},
                                  data=model.as_dict(dehydrators.group.DepartmentKindRefDehydrator))

        put_ref(ns.group, 'department.kind')
        put_ref(ns.group, 'parent.department.kind')
        put_ref(ns.group, 'ancestors.$.department.kind')

        put_ref(ns.person, 'department_group.department.kind')
        put_ref(ns.person, 'department_group.parent.department.kind')
        put_ref(ns.person, 'department_group.ancestors.$.department.kind')
        put_ref(ns.person, 'groups.$.group.department.kind')
        put_ref(ns.person, 'groups.$.group.parent.department.kind')
        put_ref(ns.person, 'groups.$.group.ancestors.$.department.kind')

        put_ref(ns.groupmembership, 'group.department.kind')
        put_ref(ns.groupmembership, 'group.parent.department.kind')
        put_ref(ns.groupmembership, 'group.encestors.$.department.kind')

    def on_delete(self, model, ns):
        ns.departmentkind.delete({'id': model.id})

        def delete_ref(collection, path):
            collection.delete_nested(path, lookup={path + '.id': model.id},
                                     set_none=None)

        delete_ref(ns.group, 'department.kind')
        delete_ref(ns.group, 'parent.department.kind')
        delete_ref(ns.group, 'ancestors.$.department.kind')

        delete_ref(ns.person, 'department_group.department.kind')
        delete_ref(ns.person, 'department_group.parent.department.kind')
        delete_ref(ns.person, 'department_group.ancestors.$.department.kind')
        delete_ref(ns.person, 'groups.$.group.department.kind')
        delete_ref(ns.person, 'groups.$.group.parent.department.kind')
        delete_ref(ns.person, 'groups.$.group.ancestors.$.department.kind')

        delete_ref(ns.groupmembership, 'group.department.kind')
        delete_ref(ns.groupmembership, 'group.parent.department.kind')
        delete_ref(ns.groupmembership, 'group.ancestors.$.department.kind')


@register('departments.departmentchain')
class DepartmentChainListener(BaseListener):
    model_class = models.group.DepartmentChainModel

    def on_modify(self, model, ns):
        data = model.as_dict(dehydrators.group.DepartmentChainDehydrator)
        ns.departmentchain.put({'id': model.id}, data=data)

        lookup = {'department_group.department.id': data['department']}
        ns.person.put_nested('hr_partners', lookup=lookup, data=data['hr_partners'])

        chiefs = data['chiefs']
        chief = chiefs[0] if chiefs else None

        lookup.update({
            'login': {'$ne': chief and chief['login']},
        })

        ns.person.put_nested('chief', lookup=lookup, data=chief)
        ns.person.put_nested('chiefs', lookup=lookup, data=chiefs)

        if chief is not None:
            lookup.update({
                'login': chief['login'],
            })
            chiefs = chiefs[1:]
            chief = chiefs[0] if chiefs else None

            ns.person.put_nested('chief', lookup=lookup, data=chief)
            ns.person.put_nested('chiefs', lookup=lookup, data=chiefs)

    def on_delete(self, model, ns):
        ns.departmentchain.delete({'id': model.id})
